<?php

namespace Drupal\doc_to_html\Services;

/**
 * Value object encapsulating the outcome of a DOC to HTML conversion.
 */
final class ConversionResult
{

  /**
   * Constructs a new conversion result.
   */
  private function __construct(
    private readonly bool $success,
    private readonly array $data = [],
    private readonly array $errors = [],
    private readonly array $warnings = [],
  ) {
  }

  /**
   * Creates a successful conversion result.
   */
  public static function success(array $data, array $warnings = []): self
  {
    return new self(TRUE, $data, [], $warnings);
  }

  /**
   * Creates a failed conversion result.
   */
  public static function failure(array $errors, array $data = [], array $warnings = []): self
  {
    return new self(FALSE, $data, $errors, $warnings);
  }

  /**
   * Indicates whether the conversion succeeded.
   */
  public function isSuccess(): bool
  {
    return $this->success;
  }

  /**
   * Returns all collected errors.
   */
  public function getErrors(): array
  {
    return $this->errors;
  }

  /**
   * Returns all collected warnings.
   */
  public function getWarnings(): array
  {
    return $this->warnings;
  }

  /**
   * Returns a single data value.
   */
  public function get(string $key, mixed $default = NULL): mixed
  {
    return $this->data[$key] ?? $default;
  }

  /**
   * Returns all data associated with the conversion.
   */
  public function getData(): array
  {
    return $this->data;
  }

}
