<?php

namespace Drupal\doc_to_html\Services;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Psr\Log\LoggerInterface;

/**
 * Provides utility features for the DOC to HTML module.
 */
class DefaultService
{

  /**
   * Constructs the default utility service.
   */
  public function __construct(
    protected ConfigFactoryInterface        $configFactory,
    protected EntityFieldManagerInterface   $entityFieldManager,
    protected EntityTypeBundleInfoInterface $bundleInfo,
    protected LoggerInterface               $logger,
  )
  {
  }

  /**
   * Returns allowed operating systems and their identifiers.
   *
   * @return array
   *   List of supported OS codes.
   */
  public function getAdmittedOS(): array
  {
    // Keys are human-readable labels, values are PHP_OS_FAMILY identifiers so
    // downstream consumers can detect the current platform quickly.
    return [
      'MAC OS' => 'Darwin',
      'LINUX' => 'Linux',
      'Windows' => 'WINNT',
    ];
  }

  /**
   * Returns default settings for a given OS.
   *
   * @param string $os
   *   OS identifier.
   *
   * @return array
   *   Array of OS-specific settings.
   */
  public function getSettingsForOS(string $os): array
  {
    return match ($os) {
      'Darwin' => [
        'base_path_application' => '/Applications/LibreOffice.app/Contents/MacOS',
        'command' => 'soffice',
      ],
      'Linux' => [
        'base_path_application' => '/usr/bin',
        'command' => 'soffice',
      ],
      'WINNT' => [
        'base_path_application' => 'C:\\Program Files\\LibreOffice\\program',
        'command' => 'soffice.exe',
      ],
      default => [
        'base_path_application' => '',
        'command' => '',
      ],
    };
  }

  /**
   * Returns list of entity+bundle+field supporting a given field type.
   *
   * @param string $fieldType
   *   The field type machine name, e.g. "file" or "text_long".
   *
   * @return array
   *   List of matching field definitions.
   */
  public function getEntityBundleFieldBy(string $fieldType): array
  {
    $result = [];

    // Ask Drupal for all fields of the given type across entity/bundle combos.
    $fieldMap = $this->entityFieldManager->getFieldMapByFieldType($fieldType);

    foreach ($fieldMap as $entityType => $fields) {
      foreach ($fields as $fieldName => $info) {
        foreach ($info['bundles'] as $bundle) {
          $definitions = $this->entityFieldManager->getFieldDefinitions($entityType, $bundle);

          if (!isset($definitions[$fieldName])) {
            continue;
          }

          // Enrich results with translated labels to drive admin UI dropdowns.
          $fieldLabel = $definitions[$fieldName]->getLabel();
          $bundleLabels = $this->bundleInfo->getBundleInfo($entityType);
          $bundleLabel = $bundleLabels[$bundle]['label'] ?? $bundle;

          $result["{$entityType}-{$bundle}-{$fieldName}"] = [
            'bundle_title' => $bundleLabel,
            'bundle' => $bundle,
            'field_label' => $fieldLabel,
            'field_name' => $fieldName,
          ];
        }
      }
    }

    return $result;
  }

  /**
   * Returns supported file extensions based on module configuration.
   *
   * @return array
   *   List of supported extensions.
   */
  public function getSupportedFileExtensions(): array
  {
    $config = $this->configFactory->get('doc_to_html.basic_settings');

    $doc = (bool)$config->get('doc');
    $docx = (bool)$config->get('docx');

    $supported = [];

    if ($doc) {
      $supported[] = 'doc';
    }

    if ($docx) {
      $supported[] = 'docx';
    }

    return $supported ?: [];
  }

}
