<?php

namespace Drupal\doc_to_html\Services;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
class FileCleaner
{
  protected EntityTypeManagerInterface $entityTypeManager;
  protected FileSystemInterface $fileSystem;

  public function __construct(EntityTypeManagerInterface $entityTypeManager, FileSystemInterface $fileSystem)
  {
    $this->entityTypeManager = $entityTypeManager;
    $this->fileSystem = $fileSystem;
  }

  /**
   * @param string $directory
   * @return int
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function cleanMissingFiles(string $directory = 'public://doc_to_html'): int
  {
    $storage = $this->entityTypeManager->getStorage('file');

    // Prendo tutti i file che hanno uri nella sotto-cartella.
    $fids = $storage->getQuery()
      ->condition('uri', $directory . '/%', 'LIKE')
      ->accessCheck(FALSE)
      ->execute();

    if (empty($fids)) {
      return 0;
    }

    $files = $storage->loadMultiple($fids);
    $deleted = 0;

    foreach ($files as $file) {
      /** @var \Drupal\file\FileInterface $file */
      $uri = $file->getFileUri();
      $realpath = $this->fileSystem->realpath($uri);

      // Se realpath è FALSE o il file non esiste fisicamente, cancello l'entità.
      if (!$realpath || !file_exists($realpath)) {
        $file->delete();
        $deleted++;
      }
    }

    return $deleted;
  }

  /**
   * @param string $directory
   * @return int
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function forceClean(string $directory = 'public://doc_to_html'): int
  {
    $deleted = 0;
    $storage = $this->entityTypeManager->getStorage('file');

    $fids = $storage->getQuery()
      ->condition('uri', $directory . '/%', 'LIKE')
      ->accessCheck(FALSE)
      ->execute();

    if (!empty($fids)) {
      $files = $storage->loadMultiple($fids);
      foreach ($files as $file) {
        $file->delete();
        $deleted++;
      }
    }

    $realpath = $this->fileSystem->realpath($directory);
    if ($realpath && is_dir($realpath)) {
      $deleted += $this->removeAllFiles($realpath);
    }

    return $deleted;
  }

  private function removeAllFiles(string $path): int
  {
    $count = 0;
    $items = @scandir($path);
    if ($items === false) {
      return 0;
    }
    foreach ($items as $item) {
      if ($item === '.' || $item === '..') {
        continue;
      }
      $full = $path . DIRECTORY_SEPARATOR . $item;
      if (is_dir($full)) {
        $count += $this->removeAllFiles($full);
        @rmdir($full);
      } elseif (is_file($full)) {
        if (@unlink($full)) {
          $count++;
        }
      }
    }
    return $count;
  }

}
