<?php

namespace Drupal\doc_to_html\Services;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\file\FileUsage\FileUsageInterface;
use Psr\Log\LoggerInterface;

/**
 * Provides file utilities for DOC to HTML conversion.
 */
class FileService implements FileServiceInterface
{

  public function __construct(
    protected ConfigFactoryInterface     $configFactory,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected FileSystemInterface        $fileSystem,
    protected FileUsageInterface         $fileUsage,
    protected LoggerInterface            $logger,
  )
  {
  }

  /**
   * {@inheritdoc}
   */
  public function cleanFolder(): void
  {
    $config = $this->configFactory->get('doc_to_html.basic_settings');
    // The folder is stored relative to public:// so this keeps behaviour in
    // sync with configuration defaults and the installer.
    $folder = $config->get('doc_to_html_folder');

    $uri = "public://{$folder}";
    $realPath = $this->fileSystem->realpath($uri);

    if (!$realPath || !is_dir($realPath)) {
      $this->logger->warning('Directory not found during cleanup: @dir', ['@dir' => $uri]);
      return;
    }

    $this->deleteManagedFiles($uri);
    $this->deleteUnmanagedFiles($realPath);
  }

  /**
   * Deletes all managed file entities within a directory.
   */
  private function deleteManagedFiles(string $uri): void
  {
    $storage = $this->entityTypeManager->getStorage('file');

    // loadByProperties with a LIKE-ish match lets us cover nested files inside
    // the conversion sub-directory.
    $files = $storage->loadByProperties([
      'uri' => $uri . '%',
    ]);

    if (!$files) {
      return;
    }

    foreach ($files as $file) {
      // Ensure file_usage is cleaned too.
      $this->fileUsage->delete($file, 'doc_to_html');
      $file->delete();
    }
  }

  /**
   * Deletes all unmanaged files in a given filesystem directory.
   */
  private function deleteUnmanagedFiles(string $path): void
  {
    $files = scandir($path);

    if (!$files) {
      return;
    }

    foreach ($files as $fileName) {
      if (in_array($fileName, ['.', '..'])) {
        continue;
      }

      $fullPath = $path . DIRECTORY_SEPARATOR . $fileName;

      if (is_file($fullPath)) {
        // Unmanaged artefacts (e.g. LibreOffice temp HTML) can be removed
        // directly from the filesystem.
        unlink($fullPath);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function realPath(string $uri): string
  {
    return $this->fileSystem->realpath($uri) ?: '';
  }

  /**
   * {@inheritdoc}
   */
  public function escapeRealPath(string $uri): string
  {
    $path = $this->realPath($uri);
    return str_replace(' ', '\\ ', $path);
  }

  /**
   * {@inheritdoc}
   */
  public function convertUriToHtml(string $uri): string
  {
    return preg_replace('/\.(docx|DOCX|doc|DOC)$/', '.html', $uri);
  }

  /**
   * {@inheritdoc}
   */
  public function prepareDirectory(string $folder): string
  {
    $folder = trim($folder, '\\/');

    if ($folder === '') {
      $this->logger->error('Attempted to prepare a directory with an empty folder name.');
      return '';
    }

    $uri = "public://{$folder}";

    $prepared = $this->fileSystem->prepareDirectory(
      $uri,
      FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS
    );

    if (!$prepared) {
      $this->logger->error('Failed to prepare directory: @dir', ['@dir' => $uri]);
      return '';
    }

    $realPath = $this->fileSystem->realpath($uri);

    if (!$realPath) {
      $this->logger->error('Unable to resolve real path for directory: @dir', ['@dir' => $uri]);
      return '';
    }

    return $realPath;
  }

}
