<?php

namespace Drupal\domain_config\Config;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Config\ConfigCollectionInfo;
use Drupal\Core\Config\ConfigCrudEvent;
use Drupal\Core\Config\ConfigFactoryOverrideBase;
use Drupal\Core\Config\ConfigRenameEvent;
use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\domain\DomainEvent;
use Drupal\domain\DomainEvents;
use Drupal\domain\DomainInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Provides domain overrides for the configuration factory.
 */
class DomainConfigFactoryOverride extends ConfigFactoryOverrideBase implements DomainConfigFactoryOverrideInterface, EventSubscriberInterface {

  use DomainConfigCollectionNameTrait;

  /**
   * The configuration storage.
   *
   * Do not access this directly. Should be accessed through self::getStorage()
   * so that the cache of storages per domain id is used.
   *
   * @var \Drupal\Core\Config\StorageInterface
   */
  protected $baseStorage;

  /**
   * An array of configuration storages keyed by domain id.
   *
   * @var \Drupal\Core\Config\StorageInterface[]
   */
  protected $storages;

  /**
   * The typed config manager.
   *
   * @var \Drupal\Core\Config\TypedConfigManagerInterface
   */
  protected $typedConfigManager;

  /**
   * The domain storage.
   *
   * @var \Drupal\domain\DomainStorageInterface
   */
  protected $domainStorage;

  /**
   * An event dispatcher instance to use for configuration events.
   *
   * @var \Symfony\Contracts\EventDispatcher\EventDispatcherInterface
   */
  protected $eventDispatcher;

  /**
   * The domain object used to override configuration data.
   *
   * @var \Drupal\domain\DomainInterface
   */
  protected $domain;

  /**
   * Constructs the DomainConfigFactoryOverride object.
   *
   * @param \Drupal\Core\Config\StorageInterface $storage
   *   The configuration storage engine.
   * @param \Symfony\Contracts\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   An event dispatcher instance to use for configuration events.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config
   *   The typed configuration manager.
   */
  public function __construct(StorageInterface $storage, EventDispatcherInterface $event_dispatcher, TypedConfigManagerInterface $typed_config) {
    $this->baseStorage = $storage;
    $this->eventDispatcher = $event_dispatcher;
    $this->typedConfigManager = $typed_config;
  }

  /**
   * Gets the domain storage.
   *
   * @return \Drupal\domain\DomainStorageInterface
   *   The domain storage handler.
   */
  protected function getDomainStorage() {
    if (!isset($this->domainStorage)) {
      // @phpstan-ignore-next-line
      $this->domainStorage = \Drupal::entityTypeManager()->getStorage('domain');
    }
    return $this->domainStorage;
  }

  /**
   * {@inheritdoc}
   */
  public function loadOverrides($names) {
    if ($this->domain) {
      $storage = $this->getStorage($this->domain->id());
      return $storage->readMultiple($names);
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getOverride($domain_id, $name) {
    $storage = $this->getStorage($domain_id);
    $data = $storage->read($name);

    $override = new DomainConfigOverride(
      $name,
      $storage,
      $this->typedConfigManager,
      $this->eventDispatcher
    );

    if (!empty($data)) {
      $override->initWithData($data);
    }
    return $override;
  }

  /**
   * {@inheritdoc}
   */
  public function getStorage($domain_id) {
    if (!isset($this->storages[$domain_id])) {
      $this->storages[$domain_id] =
        $this->baseStorage->createCollection(
          $this->createConfigCollectionName($domain_id)
        );
    }
    return $this->storages[$domain_id];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheSuffix() {
    return $this->domain ? $this->domain->id() : 'und';
  }

  /**
   * {@inheritdoc}
   */
  public function getDomain() {
    return $this->domain;
  }

  /**
   * {@inheritdoc}
   */
  public function setDomain(?DomainInterface $domain = NULL) {
    $this->domain = $domain;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function installDomainOverrides($domain_id) {
    /** @var \Drupal\Core\Config\ConfigInstallerInterface $config_installer */
    // @phpstan-ignore-next-line
    $config_installer = \Drupal::service('config.installer');
    $config_installer->installCollectionDefaultConfig($this->createConfigCollectionName($domain_id));
  }

  /**
   * {@inheritdoc}
   */
  public function createConfigObject($name, $collection = StorageInterface::DEFAULT_COLLECTION) {
    $domain_id = $this->getDomainFromCollectionName($collection);
    return $this->getOverride($domain_id, $name);
  }

  /**
   * {@inheritdoc}
   */
  public function addCollections(ConfigCollectionInfo $collection_info) {
    foreach ($this->getDomainStorage()->loadMultipleSorted() as $domain) {
      $collection_info->addCollection($this->createConfigCollectionName($domain->id()), $this);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function onConfigSave(ConfigCrudEvent $event) {
    $config = $event->getConfig();
    $name = $config->getName();
    foreach ($this->getDomainStorage()->loadMultipleSorted() as $domain) {
      $domain_config = $this->getOverride($domain->id(), $name);
      if (!$domain_config->isNew()) {
        $this->filterOverride($config, $domain_config);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function onConfigRename(ConfigRenameEvent $event) {
    $config = $event->getConfig();
    $name = $config->getName();
    $old_name = $event->getOldName();
    foreach ($this->getDomainStorage()->loadMultipleSorted() as $domain) {
      $domain_config = $this->getOverride($domain->id(), $old_name);
      if (!$domain_config->isNew()) {
        $saved_config = $domain_config->get();
        $storage = $this->getStorage($domain->id());
        $storage->write($name, $saved_config);
        $domain_config->delete();
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function onConfigDelete(ConfigCrudEvent $event) {
    $config = $event->getConfig();
    $name = $config->getName();
    foreach ($this->getDomainStorage()->loadMultipleSorted() as $domain) {
      $domain_config = $this->getOverride($domain->id(), $name);
      if (!$domain_config->isNew()) {
        $domain_config->delete();
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheableMetadata($name) {
    $metadata = new CacheableMetadata();
    if ($this->domain) {
      $metadata->setCacheContexts(['domain']);
    }
    return $metadata;
  }

  /**
   * {@inheritdoc}
   */
  public function onActiveDomainSet(DomainEvent $event) {
    $this->setDomain($event->getDomain());
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events = parent::getSubscribedEvents();
    $events[DomainEvents::ACTIVE_DOMAIN_SET][] = ['onActiveDomainSet'];
    return $events;
  }

}
