<?php

namespace Drupal\domain_config_ui\Config;

use Drupal\Core\Config\ConfigCrudEvent;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\domain\DomainEvent;
use Drupal\domain\DomainEvents;
use Drupal\domain_config\Config\DomainConfigFactoryOverride;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Extends core ConfigFactory class to save domain specific configuration.
 */
class DomainConfigFactory extends ConfigFactory {

  const DOMAIN_CONFIG_UI_DISALLOWED_CONFIGURATIONS = [
    'domain_config_ui.settings',
    'domain.settings',
    'language.types',
  ];

  /**
   * The domain overriding config factory service.
   *
   * @var \Drupal\domain_config\Config\DomainConfigFactoryOverride
   */
  protected DomainConfigFactoryOverride $overrideFactory;

  /**
   * The current route match service.
   *
   * @var \Drupal\Core\Routing\CurrentRouteMatch
   */
  protected $currentRouteMatch;

  /**
   * Drupal\Core\Routing\AdminContext definition.
   *
   * @var \Drupal\Core\Routing\AdminContext
   */
  protected $adminContext;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The domain object used to override configuration data.
   *
   * @var \Drupal\domain\DomainInterface
   */
  protected $domain;

  /**
   * The overridable configurations.
   *
   * @var array
   */
  protected $overridableConfigurations;

  /**
   * List of configuration names that must not be overridden.
   *
   * @var array|null
   */
  protected $disallowedConfigurations = NULL;

  /**
   * Constructs the Config factory.
   *
   * @param \Drupal\Core\Config\StorageInterface $storage
   *   The configuration storage engine.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   An event dispatcher instance to use for configuration events.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config
   *   The typed configuration manager.
   * @param \Drupal\domain_config\Config\DomainConfigFactoryOverride $override_factory
   *   The domain config override factory.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   */
  public function __construct(
    StorageInterface $storage,
    EventDispatcherInterface $event_dispatcher,
    TypedConfigManagerInterface $typed_config,
    DomainConfigFactoryOverride $override_factory,
    ModuleHandlerInterface $module_handler,
  ) {
    parent::__construct($storage, $event_dispatcher, $typed_config);
    $this->overrideFactory = $override_factory;
    $this->moduleHandler = $module_handler;
  }

  /**
   * Create a domain editable configuration object.
   *
   * @param string $name
   *   The name of the configuration object to create.
   *
   * @return \Drupal\Core\Config\Config
   *   A new configuration object that is editable per domain.
   */
  protected function createDomainEditableConfigObject($name) {
    $storage = $this->getDomainConfigStorage();
    return new Config($name, $storage, $this->eventDispatcher, $this->typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  protected function doLoadMultiple(array $names, $immutable = TRUE) {
    // Don't override config if immutable or not per domain editable.
    if ($immutable || !$this->isPerDomainEditable($names)) {
      return parent::doLoadMultiple($names, $immutable);
    }

    $list = [];

    foreach ($names as $key => $name) {
      $cache_key = $this->getDomainEditableConfigCacheKey($name);
      if (isset($this->cache[$cache_key])) {
        $list[$name] = $this->cache[$cache_key];
        unset($names[$key]);
      }
    }

    // Pre-load remaining configuration files.
    if ($names !== []) {
      $storage_data = $this->getDomainConfigStorage()->readMultiple($names);
      foreach ($names as $name) {
        // Fetch base config data if not yet overridden.
        $data = $storage_data[$name] ?? $this->storage->read($name);
        $cache_key = $this->getDomainEditableConfigCacheKey($name);
        $this->cache[$cache_key] = $this->createDomainEditableConfigObject($name);
        $this->cache[$cache_key]->initWithData($data);
        $list[$name] = $this->cache[$cache_key];
      }
    }

    return $list;
  }

  /**
   * {@inheritdoc}
   */
  protected function doGet($name, $immutable = TRUE) {
    // Don't override config if immutable or not per domain editable.
    if ($immutable || !$this->isPerDomainEditable($name)) {
      return parent::doGet($name, $immutable);
    }
    $config = $this->doLoadMultiple([$name], $immutable);
    if (isset($config[$name])) {
      return $config[$name];
    }
    else {
      // If the configuration object does not exist in the configuration
      // storage, create a new object.
      return $this->createDomainEditableConfigObject($name);
    }
  }

  /**
   * Get the cache key for a domain editable configuration object.
   *
   * @param string $name
   *   The name of the configuration object.
   *
   * @return string
   *   The cache key for the domain configuration object.
   */
  protected function getDomainEditableConfigCacheKey($name) {
    // We want to be able to cache all editable domain-specific configuration
    // objects, so we need to include the domain cache keys in the cache key.
    // Default implementation only add cache keys to immutable config.
    $suffix = ':' . $this->getActiveDomainId();
    // To avoid potential conflicts with the default config cache key.
    $suffix .= ':editable';
    return $name . $suffix;
  }

  /**
   * Get the configuration storage for the current domain.
   *
   * @return \Drupal\Core\Config\StorageInterface
   *   The configuration storage for the current domain.
   */
  protected function getDomainConfigStorage() {
    $domain_id = $this->getActiveDomainId();
    return $this->overrideFactory->getStorage($domain_id);
  }

  /**
   * Get the selected domain ID.
   *
   * @return string|null
   *   A domain machine name.
   */
  protected function getActiveDomainId() {
    return $this->domain ? $this->domain->id() : NULL;
  }

  /**
   * Check if a selected domain is available in request or session.
   */
  protected function hasActiveDomainId() {
    return !is_null($this->domain);
  }

  /**
   * Get the configured overridable configurations.
   *
   * @return array
   *   The overridable configurations.
   */
  public function getOverridableConfigurations() {
    if (!isset($this->overridableConfigurations)) {
      $config = $this->get('domain_config_ui.settings');
      $overridable_configurations = $config->get('overridable_configurations');
      $this->overridableConfigurations = [];
      if ($overridable_configurations) {
        foreach ($overridable_configurations as $configuration) {
          $this->overridableConfigurations[$configuration['name']] = $configuration['domains'];
        }
      }
    }
    return $this->overridableConfigurations;
  }

  /**
   * Checks if a domain configuration is allowed to be overridden.
   *
   * @param string $names
   *   A configuration name.
   *
   * @return bool
   *   TRUE if configuration is overridable per domain, FALSE otherwise.
   */
  public function isRegisteredConfiguration($names) {
    $overridable_configurations = $this->getOverridableConfigurations();
    $name = is_array($names) ? current($names) : $names;
    if (isset($overridable_configurations[$name])) {
      return in_array($this->getActiveDomainId(), $overridable_configurations[$name]);
    }
    return FALSE;
  }

  /**
   * Check that a specific config can be edited per domain.
   *
   * @param string|array $names
   *   The config name.
   *
   * @return bool
   *   TRUE if it can be edited by domain, FALSE otherwise.
   */
  public function isAllowedConfiguration($names):bool {
    if (!isset($this->disallowedConfigurations)) {
      // Never allow this module's settings to be added, for example.
      $this->disallowedConfigurations = static::DOMAIN_CONFIG_UI_DISALLOWED_CONFIGURATIONS;
      // Allow modules to alter the list of disallowed configurations.
      $this->moduleHandler->alter('domain_config_ui_disallowed_configurations', $this->disallowedConfigurations);
    }
    if (is_array($names)) {
      if (!empty(array_intersect($names, $this->disallowedConfigurations))) {
        return FALSE;
      }
    }
    else {
      if (in_array($names, $this->disallowedConfigurations, TRUE)) {
        return FALSE;
      }
    }
    return TRUE;
  }

  /**
   * Checks if the current route and path are domain-configurable.
   *
   * @param string|array $names
   *   The config name.
   *
   * @return bool
   *   TRUE if domain-configurable, false otherwise.
   */
  public function isPerDomainEditable($names) {
    return $this->hasActiveDomainId()
      && $this->isRegisteredConfiguration($names)
      && $this->isAllowedConfiguration($names);
  }

  /**
   * Reacts to the DomainEvents::ACTIVE_DOMAIN_SET event.
   *
   * @param \Drupal\domain\DomainEvent $event
   *   The domain negotiated event.
   */
  public function onActiveDomainSet(DomainEvent $event) {
    $this->domain = $event->getDomain();
  }

  /**
   * {@inheritdoc}
   */
  public function onConfigSave(ConfigCrudEvent $event): void {
    parent::onConfigSave($event);
    $config = $event->getConfig();
    if ($config->getName() === 'domain_config_ui.settings') {
      $this->overridableConfigurations = NULL;
      $this->getOverridableConfigurations();
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events = parent::getSubscribedEvents();
    $events[DomainEvents::ACTIVE_DOMAIN_SET][] = ['onActiveDomainSet'];
    return $events;
  }

}
