<?php

namespace Drupal\domain_config_ui;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Routing\AdminContext;
use Drupal\Core\Routing\CurrentRouteMatch;
use Drupal\domain\DomainEvent;
use Drupal\domain\DomainEvents;
use Drupal\domain_config_ui\Config\DomainConfigFactory;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Domain Config UI manager.
 */
class DomainConfigUIManager implements DomainConfigUIManagerInterface, EventSubscriberInterface {

  const DOMAIN_CONFIG_UI_DISALLOWED_ROUTES = [
    'domain_config_ui.settings',
    'domain.settings',
  ];

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The current route match service.
   *
   * @var \Drupal\Core\Routing\CurrentRouteMatch
   */
  protected $currentRouteMatch;

  /**
   * Drupal\Core\Routing\AdminContext definition.
   *
   * @var \Drupal\Core\Routing\AdminContext
   */
  protected $adminContext;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * List of route names that should not allow overrides.
   *
   * @var array|null
   */
  protected $disallowedRoutes = NULL;

  /**
   * TRUE if the current page is an admin route.
   *
   * @var bool
   */
  protected $adminRoute;

  /**
   * The domain object used to override configuration data.
   *
   * @var \Drupal\domain\DomainInterface
   */
  protected $domain;

  /**
   * Constructs DomainConfigUIManager object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Configuration service.
   * @param \Drupal\Core\Routing\CurrentRouteMatch $current_route_match
   *   The current route match service.
   * @param \Drupal\Core\Routing\AdminContext $admin_context
   *   The admin context.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    CurrentRouteMatch $current_route_match,
    AdminContext $admin_context,
    ModuleHandlerInterface $module_handler,
  ) {
    $this->configFactory = $config_factory;
    $this->currentRouteMatch = $current_route_match;
    $this->adminContext = $admin_context;
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public function isAdminRoute() {
    if (!isset($this->adminRoute)) {
      $admin_route = $this->checkAllowedRoute();
      if (is_bool($admin_route)) {
        $this->adminRoute = $admin_route;
      }
      else {
        // Route probably not yet available.
        return FALSE;
      }
    }
    return $this->adminRoute;
  }

  /**
   * Checks if the route is allowed and is an admin route.
   *
   * @return bool|null
   *   TRUE if the route is allowed, FALSE otherwise. NULL if undefined.
   */
  protected function checkAllowedRoute() {
    $route_name = $this->currentRouteMatch->getRouteName();
    if (is_null($route_name)) {
      return NULL;
    }
    if (!isset($this->disallowedRoutes)) {
      // Never allow this module's form to be added.
      $this->disallowedRoutes = self::DOMAIN_CONFIG_UI_DISALLOWED_ROUTES;
      // Allow modules to alter the list of disallowed routes.
      $this->moduleHandler->alter('domain_config_ui_disallowed_routes', $this->disallowedRoutes);
    }
    if (in_array($route_name, $this->disallowedRoutes, TRUE)) {
      return FALSE;
    }
    $route = $this->currentRouteMatch->getRouteObject();
    return $this->adminContext->isAdminRoute($route);
  }

  /**
   * {@inheritdoc}
   */
  public function isAllowedConfiguration($names):bool {
    if ($this->configFactory instanceof DomainConfigFactory) {
      return $this->configFactory->isAllowedConfiguration($names);
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function isRegisteredConfiguration($names) {
    if ($this->configFactory instanceof DomainConfigFactory) {
      return $this->configFactory->isRegisteredConfiguration($names);
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function addConfigurationToDomain($domain_id, $config_name) {
    $success = TRUE;
    $config = $this->configFactory->getEditable('domain_config_ui.settings');
    $overridable_configurations = $config->get('overridable_configurations') ?? [];
    $config_index = NULL;
    // Check to see if we already registered this form.
    foreach ($overridable_configurations as $index => $configuration) {
      if ($configuration['name'] === $config_name) {
        $config_index = $index;
        if (in_array($domain_id, $configuration['domains'])) {
          $success = FALSE;
          break;
        }
      }
    }
    if ($success) {
      if (is_null($config_index)) {
        $overridable_configurations[] = ['name' => $config_name, 'domains' => [$domain_id]];
      }
      else {
        $overridable_configurations[$config_index]['domains'][] = $domain_id;
      }
      $config->set('overridable_configurations', $overridable_configurations);
      $config->save();
    }
    return $success;
  }

  /**
   * {@inheritdoc}
   */
  public function removeConfigurationFromDomain($domain_id, $config_name) {
    $success = FALSE;
    $config = $this->configFactory->getEditable('domain_config_ui.settings');
    $overridable_configurations = $config->get('overridable_configurations');
    $config_key = NULL;
    foreach ($overridable_configurations as $key => $configuration) {
      if ($configuration['name'] === $config_name) {
        if (in_array($domain_id, $configuration['domains'])) {
          $config_key = $key;
          $success = TRUE;
        }
        break;
      }
    }
    if ($success) {
      $domain_key = array_search($domain_id, $overridable_configurations[$config_key]['domains']);
      unset($overridable_configurations[$config_key]['domains'][$domain_key]);
      if (empty($overridable_configurations[$config_key]['domains'])) {
        unset($overridable_configurations[$config_key]);
      }
      $config->set('overridable_configurations', $overridable_configurations);
      $config->save();
    }
    return $success;
  }

  /**
   * Get the selected domain ID.
   *
   * @return string|null
   *   A domain machine name.
   */
  protected function getActiveDomainId() {
    return $this->domain ? $this->domain->id() : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function addConfigurationToCurrentDomain($config_name) {
    $domain_id = $this->getActiveDomainId();
    return $this->addConfigurationToDomain($domain_id, $config_name);
  }

  /**
   * {@inheritdoc}
   */
  public function removeConfigurationFromCurrentDomain($config_name) {
    $domain_id = $this->getActiveDomainId();
    return $this->removeConfigurationFromDomain($domain_id, $config_name);
  }

  /**
   * Reacts to the DomainEvents::ACTIVE_DOMAIN_SET event.
   *
   * @param \Drupal\domain\DomainEvent $event
   *   The domain negotiated event.
   */
  public function onActiveDomainSet(DomainEvent $event) {
    $this->domain = $event->getDomain();
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events = [];
    $events[DomainEvents::ACTIVE_DOMAIN_SET][] = ['onActiveDomainSet'];
    return $events;
  }

}
