<?php

namespace Drupal\domain_config_ui;

use Drupal\domain\DomainEvent;

/**
 * Domain Config UI manager.
 */
interface DomainConfigUIManagerInterface {

  /**
   * Checks if route is admin.
   *
   * @return bool
   *   TRUE if route is admin. Otherwise, FALSE.
   */
  public function isAdminRoute();

  /**
   * Check that a specific config can be edited per domain.
   *
   * @param string|array $names
   *   The config name.
   *
   * @return bool
   *   TRUE if it can be edited by domain, FALSE otherwise.
   */
  public function isAllowedConfiguration($names):bool;

  /**
   * Checks if a domain configuration is allowed to be overridden.
   *
   * @param string $names
   *   A configuration name.
   *
   * @return bool
   *   TRUE if configuration is overridable per domain, FALSE otherwise.
   */
  public function isRegisteredConfiguration($names);

  /**
   * Add configuration to a specific domain.
   *
   * @param string $domain_id
   *   The domain id.
   * @param string $config_name
   *   A configuration name.
   *
   * @return bool
   *   TRUE if successfully added, FALSE otherwise.
   */
  public function addConfigurationToDomain($domain_id, $config_name);

  /**
   * Remove configuration from a specific domain.
   *
   * @param string $domain_id
   *   The domain id.
   * @param string $config_name
   *   A configuration name.
   *
   * @return bool
   *   TRUE if successfully removed, FALSE otherwise.
   */
  public function removeConfigurationFromDomain($domain_id, $config_name);

  /**
   * Add configuration to the current domain.
   *
   * @param string $config_name
   *   A configuration name.
   *
   * @return bool
   *   TRUE if successfully added, FALSE otherwise.
   */
  public function addConfigurationToCurrentDomain($config_name);

  /**
   * Remove configuration from current domain.
   *
   * @param string $config_name
   *   A configuration name.
   *
   * @return bool
   *   TRUE if successfully removed, FALSE otherwise.
   */
  public function removeConfigurationFromCurrentDomain($config_name);

  /**
   * Reacts to the DomainEvents::ACTIVE_DOMAIN_SET event.
   *
   * @param \Drupal\domain\DomainEvent $event
   *   The domain negotiated event.
   */
  public function onActiveDomainSet(DomainEvent $event);

}
