<?php

namespace Drupal\domain_config\Service;

/**
 * @file
 * Migration path from Domain module 2.x to 3.x configuration collections.
 */

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\domain\DomainInterface;
use Drupal\domain_config\Config\DomainConfigCollectionUtils;

/**
 * Service class for migrating Domain 2.x configurations to 3.x collections.
 */
class DomainConfigMigration {

  /**
   * The list of domains.
   *
   * @var \Drupal\domain\DomainInterface[]
   */
  protected $domains;

  /**
   * The list of languages.
   *
   * @var \Drupal\Core\Language\LanguageInterface[]
   */
  protected $languages;

  /**
   * The default language.
   *
   * @var \Drupal\Core\Language\LanguageInterface
   */
  protected $defaultLanguage;

  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected StorageInterface $configStorage,
    protected TypedConfigManagerInterface $typedConfigManager,
    EntityTypeManagerInterface $entity_type_manager,
    LanguageManagerInterface $language_manager,
  ) {
    $domain_storage = $entity_type_manager->getStorage('domain');
    $this->domains = $domain_storage->loadMultiple();
    $this->languages = $language_manager->getLanguages();
    $this->defaultLanguage = $language_manager->getDefaultLanguage();
  }

  /**
   * Migrate domain configurations from 2.x to 3.x format.
   *
   * @return array
   *   Migration results with success/failure counts.
   */
  public function migrateConfigurations(): array {
    $results = [
      'migrated' => [],
      'failed' => 0,
      'errors' => [],
    ];

    $config = $this->configFactory->getEditable('domain_config_ui.settings');
    $overridable_configurations = $config->get('overridable_configurations') ?? [];

    $overridable_configurations_map = $this->toConfigurationMap($overridable_configurations);

    foreach ($this->domains as $domain) {
      try {
        $this->migrateDomainConfiguration(
          $domain, $overridable_configurations_map, $results,
        );
      }
      catch (\Exception $e) {
        $results['failed']++;
        $results['errors'][] = sprintf(
          'Failed to migrate domain %s: %s',
          $domain['machine_name'],
          $e->getMessage()
        );
      }
    }

    // Clean up migrated legacy configuration files.
    $this->cleanupLegacyConfigurations($results['migrated']);

    $overridable_configurations =
      $this->fromConfigurationMap($overridable_configurations_map);

    $config->set('overridable_configurations', $overridable_configurations);

    // Save updated configuration.
    $config->save();

    return $results;
  }

  /**
   * Migrate a single domain's configuration overrides to collection format.
   *
   * @param \Drupal\domain\DomainInterface $domain
   *   The domain entity.
   * @param array $overridable_configurations
   *   The list of overridable configurations.
   * @param array $results
   *   The migration results.
   */
  protected function migrateDomainConfiguration(
    DomainInterface $domain,
    array &$overridable_configurations,
    array &$results,
  ): void {
    $domain_id = $domain->id();

    // Get legacy configuration overrides for this domain.
    $legacy_overrides = $this->configStorage->listAll("domain.config.{$domain_id}.");
    if (empty($legacy_overrides)) {
      return;
    }

    // Create a configuration collection for this domain.
    $domain_collection_name =
      DomainConfigCollectionUtils::createDomainConfigCollectionName($domain_id);
    $domain_collection_storage = $this->configStorage->createCollection($domain_collection_name);
    // Create configuration collections for this domain and all languages.
    $domain_languages_collection_storage = [];
    foreach ($this->languages as $language) {
      $domain_language_collection_name =
        DomainConfigCollectionUtils::createDomainLanguageConfigCollectionName(
          $domain_id, $language->getId());
      $domain_languages_collection_storage[$language->getId()] =
        $this->configStorage->createCollection($domain_language_collection_name);
    }

    $pattern = '/^domain\.config\.' . $domain_id . '(?:\.([a-z]{2}))?\.([^.]+\.[^.]+)$/';

    // Migrate each override to the collection.
    foreach ($legacy_overrides as $legacy_name) {
      if (preg_match($pattern, $legacy_name, $matches)) {
        // $matches[1] = language code (if present)
        // $matches[2] = config name
        $lang = $matches[1] ?? NULL;
        $name = $matches[2];
        $overridable_configurations[$name][$domain_id] = $domain_id;

        // Write override to the corresponding collection.
        $config_data = $this->configStorage->read($legacy_name);
        if (!$config_data) {
          continue;
        }

        if (empty($lang)) {
          $domain_collection_storage->write($name, $config_data);
          $results['migrated'][] = $legacy_name;
        }
        else {
          if (in_array($lang, array_keys($this->languages), TRUE)) {
            $domain_languages_collection_storage[$lang]->write($name, $config_data);
            $results['migrated'][] = $legacy_name;
          }
        }
      }
    }

  }

  /**
   * Removes legacy domain configuration overrides after migration.
   *
   * @param array $legacy_names
   *   The list of legacy configuration overrides to delete.
   *
   * @return int
   *   The number of legacy configuration overrides deleted.
   */
  protected function cleanupLegacyConfigurations(array $legacy_names): int {
    // Clean up legacy domain config overrides.
    $total_deleted = 0;
    foreach ($legacy_names as $legacy_name) {
      if ($this->configStorage->delete($legacy_name)) {
        $total_deleted += 1;
      }
    }
    return $total_deleted;
  }

  /**
   * Removes legacy domain configuration overrides after migration.
   *
   * @return int
   *   The number of legacy configuration overrides deleted.
   */
  public function cleanupAllLegacyConfigurations(): int {
    $legacy_names = $this->configStorage->listAll('domain.config.');
    return $this->cleanupLegacyConfigurations($legacy_names);
  }

  /**
   * Convert the configuration array to a map.
   */
  private function toConfigurationMap(array $configurations): array {
    $map = [];
    foreach ($configurations as $configuration) {
      $map[$configuration['name']] =
        array_combine($configuration['domains'], $configuration['domains']);
    }
    return $map;
  }

  /**
   * Convert the configuration map to an array.
   */
  private function fromConfigurationMap(array $configurations): array {
    $array = [];
    foreach ($configurations as $name => $domains) {
      $array[] = [
        'name' => $name,
        'domains' => array_keys($domains),
      ];
    }
    return $array;
  }

}
