<?php

namespace Drupal\Tests\domain_access\Functional;

use Drupal\Core\Url;
use Drupal\field\Entity\FieldConfig;
use Drupal\language\Entity\ContentLanguageSettings;
use Drupal\Tests\domain\Functional\DomainTestBase;
use Drupal\domain_access\DomainAccessManager;
use Drupal\domain_access\DomainAccessManagerInterface;
use Drupal\language\Entity\ConfigurableLanguage;

/**
 * Tests saving the domain access field elements in multiple languages.
 *
 * @group domain_access
 */
class DomainAccessLanguageSaveTest extends DomainTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'domain',
    'domain_access',
    'field',
    'user',
    'language',
    'content_translation',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create 5 domains.
    $this->domainCreateTestDomains(5);

    // Add Hungarian and Afrikaans.
    ConfigurableLanguage::createFromLangcode('hu')->save();
    ConfigurableLanguage::createFromLangcode('af')->save();

    // Enable content translation for the current entity type.
    \Drupal::service('content_translation.manager')->setEnabled('node', 'page', TRUE);
  }

  /**
   * Basic test setup.
   */
  public function testDomainAccessSave() {
    $storage = \Drupal::entityTypeManager()->getStorage('node');
    // Save a node programmatically.
    $node = $storage->create([
      'type' => 'article',
      'title' => 'Test node',
      'uid' => '1',
      'status' => 1,
      DomainAccessManagerInterface::DOMAIN_ACCESS_FIELD => ['example_com'],
      DomainAccessManagerInterface::DOMAIN_ACCESS_ALL_FIELD => 1,
    ]);
    $node->save();

    // Load the node.
    $node = $storage->load(1);

    // Check that two values are set properly.
    $values = DomainAccessManager::getAccessValues($node);
    $this->assertCount(1, $values, 'Node saved with one domain records.');
    $value = DomainAccessManager::getAllValue($node);
    $this->assertTrue($value, 'Node saved to all affiliates.');

    // Create an Afrikaans translation assigned to domain 2.
    $translation = $node->addTranslation('af');
    $translation->set('title', $this->randomString());
    $translation->set(DomainAccessManagerInterface::DOMAIN_ACCESS_FIELD, [
      'example_com',
      'one_example_com',
    ]);
    $translation->set(DomainAccessManagerInterface::DOMAIN_ACCESS_ALL_FIELD, 0);
    $translation->set('status', 1);
    $node->save();

    // Load and check the translated node.
    $parent_node = $storage->load(1);
    $node = $parent_node->getTranslation('af');
    $values = DomainAccessManager::getAccessValues($node);
    $this->assertCount(2, $values, 'Node saved with two domain records.');
    $value = DomainAccessManager::getAllValue($node);
    $this->assertFalse($value, 'Node not saved to all affiliates.');
  }

  /**
   * Tests saving domain access fields when creating translations.
   *
   * This test verifies that domain access fields are handled correctly
   * when a node is translated, including the assignment of domains
   * and the behavior of untranslatable fields.
   */
  public function testDomainAccessSaveTranslation() {
    $editorRole = $this->drupalCreateRole([
      'publish to any assigned domain',
      'create domain content',
      'edit domain content',
      'delete domain content',
      'edit any page content',
      'create page content',
      'create content translations',
      'delete content translations',
      'update content translations',
      'translate editable entities',
    ]);
    $editor1 = $this->createUser();
    $editor1->addRole($editorRole);
    $editor1->save();
    $this->addDomainsToEntity('user', $editor1->id(), ['one_example_com', 'two_example_com'], DomainAccessManagerInterface::DOMAIN_ACCESS_FIELD);

    $editor2 = $this->createUser();
    $editor2->addRole($editorRole);
    $editor2->save();
    $this->addDomainsToEntity('user', $editor2->id(), ['two_example_com'], DomainAccessManagerInterface::DOMAIN_ACCESS_FIELD);

    FieldConfig::loadByName('node', 'page', DomainAccessManagerInterface::DOMAIN_ACCESS_FIELD)
      ->set('translatable', FALSE)
      ->save();

    $contentLanguageSettings = ContentLanguageSettings::loadByEntityTypeBundle('node', 'page');
    $settings = $contentLanguageSettings->getThirdPartySetting('content_translation', 'bundle_settings');
    $settings['untranslatable_fields_hide'] = '1';
    $contentLanguageSettings->setThirdPartySetting('content_translation', 'bundle_settings', $settings);
    $contentLanguageSettings->save();

    $storage = \Drupal::entityTypeManager()->getStorage('node');

    // Save a node programmatically.
    $node = $storage->create([
      'type' => 'page',
      'title' => 'Test node',
      'uid' => '1',
      'status' => 1,
      DomainAccessManagerInterface::DOMAIN_ACCESS_FIELD => ['one_example_com'],
      DomainAccessManagerInterface::DOMAIN_ACCESS_ALL_FIELD => 0,
    ]);
    $node->save();

    $this->drupalLogin($editor1);
    $this->drupalGet($node->toUrl('edit-form'));
    $edit = [
      'field_domain_access[one_example_com]' => 'one_example_com',
      'field_domain_access[two_example_com]' => 'two_example_com',
    ];
    $this->submitForm($edit, 'Save');

    // Login with editor2 and create translations.
    $this->drupalLogin($editor2);
    $this->drupalGet(Url::fromRoute('entity.node.content_translation_overview', ['node' => $node->id()]));
    $this->drupalGet('hu/node/' . $node->id() . '/translations/add/en/hu');
    $this->submitForm([], 'Save');
    $this->assertSession()->addressEquals('hu/node/1');

    // Clear the local static cache to reflect the remote changes.
    DomainAccessManager::clearStaticCache();

    /** @var \Drupal\node\NodeInterface $node */
    $node = $storage->load($node->id());
    $node_translation = $node->getTranslation('hu');
    $this->assertEquals(2, $node_translation->get(DomainAccessManagerInterface::DOMAIN_ACCESS_FIELD)->count());
    $this->assertCount(2, DomainAccessManager::getAccessValues($node_translation));
  }

}
