<?php

namespace Drupal\domain_config\Config;

use Drupal\Core\Config\ConfigFactoryOverrideInterface;
use Drupal\domain\DomainEvent;
use Drupal\domain\DomainInterface;

/**
 * Defines the interface for a configuration factory domain override object.
 */
interface DomainConfigFactoryOverrideInterface extends ConfigFactoryOverrideInterface {

  /**
   * Gets the domain object used to override configuration data.
   *
   * @return \Drupal\domain\DomainInterface
   *   The domain object used to override configuration data.
   */
  public function getDomain();

  /**
   * Sets the domain to be used in configuration overrides.
   *
   * @param \Drupal\domain\DomainInterface $domain
   *   The domain object used to override configuration data.
   *
   * @return $this
   */
  public function setDomain(?DomainInterface $domain = NULL);

  /**
   * Get override for a given domain and configuration name.
   *
   * @param string $domain_id
   *   Domain ID.
   * @param string $name
   *   Configuration name.
   *
   * @return \Drupal\Core\Config\Config
   *   Configuration override object.
   */
  public function getOverride($domain_id, $name);

  /**
   * Get override for a given domain and configuration name.
   *
   * @param string $domain_id
   *   Domain ID.
   * @param string $name
   *   Configuration name.
   *
   * @return \Drupal\Core\Config\Config
   *   Configuration override editable object.
   */
  public function getOverrideEditable($domain_id, $name);

  /**
   * Returns the storage instance for a particular domain.
   *
   * @param string $domain_id
   *   Domain ID.
   *
   * @return \Drupal\Core\Config\StorageInterface
   *   The storage instance for a particular domain.
   */
  public function getStorage($domain_id);

  /**
   * Installs available configuration overrides for a given domain.
   *
   * @param string $domain_id
   *   Domain ID.
   */
  public function installDomainOverrides($domain_id);

  /**
   * Reacts to the DomainEvents::ACTIVE_DOMAIN_SET event.
   *
   * @param \Drupal\domain\DomainEvent $event
   *   The domain negotiated event.
   */
  public function onActiveDomainSet(DomainEvent $event);

}
