<?php

namespace Drupal\domain_boolean\Plugin\Field\FieldFormatter;

use Drupal\Core\Entity\EntityTypeManager;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;

/**
 * Plugin implementation of the 'domain_boolean_formatter' formatter.
 *
 * @FieldFormatter(
 *   id = "domain_boolean_formatter",
 *   label = @Translation("Domain Boolean"),
 *   field_types = {
 *     "domain_boolean_field"
 *   }
 * )
 */
class DomainBooleanFormatter extends FormatterBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManager
   */
  protected $entityTypeManager;

  /**
   * Constructs a new DomainBooleanFormatter object.
   *
   * @param string $plugin_id
   *   The plugin ID for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\Core\Entity\EntityTypeManager $entity_type_manager
   *   The entity type manager service.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, $label, $view_mode, array $third_party_settings, EntityTypeManager $entity_type_manager) {
    // Call the parent constructor with the correct arguments.
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    // Store your injected service.
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      // Pass the service to the constructor.
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    // Your viewElements implementation, which can now use $this->domainNegotiator
    $elements = [];
    $all_domains = $this->entityTypeManager->getStorage('domain')->loadMultiple();

    // Use the standard field label.
    $field_label = $this->fieldDefinition->getLabel();

    foreach ($items as $delta => $item) {
      $featured_domains = [];
      $domain_statuses = $item->value;

      if (!empty($domain_statuses) && is_array($domain_statuses)) {
        foreach ($domain_statuses as $domain_id => $is_featured) {
          if ($is_featured && isset($all_domains[$domain_id])) {
            $featured_domains[] = $all_domains[$domain_id]->label();
          }
        }
      }

      if (!empty($featured_domains)) {
        $elements[$delta] = [
          '#theme' => 'item_list',
          // Use the standard field label for the list title.
          '#title' => $field_label,
          '#items' => $featured_domains,
          '#empty' => $this->t('Not featured on any domain.'),
        ];
      }
    }
    return $elements;
  }
}
