<?php

namespace Drupal\domain_boolean\Plugin\Field\FieldWidget;

use Drupal\Core\Entity\EntityTypeManager;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;

/**
 * Plugin implementation of the 'domain_boolean_widget' widget.
 *
 * @FieldWidget(
 *   id = "domain_boolean_widget",
 *   label = @Translation("Domain Boolean Checkboxes"),
 *   field_types = {
 *     "domain_boolean_field"
 *   }
 * )
 */
class DomainBooleanWidget extends WidgetBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManager
   */
  protected $entityTypeManager;

  /**
   * Constructs a new DomainBooleanWidget object.
   *
   * @param string $plugin_id
   *   The plugin_id for the widget.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the widget is associated.
   * @param array $settings
   *   The widget settings.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\Core\Entity\EntityTypeManager $entity_type_manager
   *   The entity type manager service.
 */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, array $third_party_settings, EntityTypeManager $entity_type_manager) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element = [];

    // Get all active domains.
    $all_domains = $this->entityTypeManager->getStorage('domain')->loadMultiple();
    $options = [];
    foreach ($all_domains as $domain_id => $domain) {
      $options[$domain_id] = $domain->label();
    }
    $current_values = $items[$delta]->value ?? [];

    // Use the standard field label and description.
    $field_label = $this->fieldDefinition->getLabel();
    $field_description = $this->fieldDefinition->getDescription();

    $element['value'] = [
      '#type' => 'checkboxes',
      // Use the standard field label.
      '#title' => $field_label,
      '#options' => $options,
      '#default_value' => array_keys(array_filter($current_values)),
      // Use the standard field description.
      '#description' => $field_description,
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    // The checkboxes return an array where unchecked items have a value of 0.
    // We need to convert this into our desired format:
    // ['domain_id' => TRUE/FALSE].
    $massaged_values = [];
    foreach ($values as $delta => $item_values) {
      $domain_statuses = [];
      if (isset($item_values['value'])) {
        foreach ($item_values['value'] as $domain_id => $status) {
          // If $status is not 0 (i.e., it's the domain ID), it means it was checked.
          $domain_statuses[$domain_id] = ($status !== 0);
        }
      }
      $massaged_values[$delta]['value'] = $domain_statuses;
    }
    return $massaged_values;
  }

}
