<?php

namespace Drupal\domain_config_extras;

use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\domain_config\DomainConfigOverrider;

/**
 * Auxiliary functions for operating with domains and configuration overrides.
 */
class DomainConfigUtilities {

  /**
   * The domain entity storage.
   *
   * @var \Drupal\domain\DomainStorageInterface
   */
  protected $domainStorage;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The configuration entity storage.
   *
   * @var \Drupal\Core\Config\StorageInterface
   */
  protected $configStorage;

  /**
   * Constructs a DomainConfigUtilities object.
   *
   * @param \Drupal\Core\Config\StorageInterface $config_storage
   *   The configuration storage.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   */
  public function __construct(StorageInterface $config_storage, EntityTypeManagerInterface $entity_type_manager, LanguageManagerInterface $language_manager) {
    $this->domainStorage = $entity_type_manager->getStorage('domain');
    $this->languageManager = $language_manager;
    $this->configStorage = $config_storage;
  }

  /**
   * Return config overrides across all (or only active) domains.
   *
   * @param array $names
   *   A list of configuration names that are being loaded.
   * @param bool $only_active
   *   Whether to load overrides only for active domains or all.
   *
   * @return array
   *   An array keyed by domain name, then configuration name of override data.
   *   Override data contains a nested array structure of overrides.
   */
  public function loadAllDomainOverrides(array $names, $only_active = FALSE): array {
    $result = [];

    if ($only_active) {
      $domains = $this->domainStorage->loadByProperties(['active' => 1]);
    }
    else {
      $domains = $this->domainStorage->loadMultiple();
    }
    $languages = $this->languageManager->getLanguages();

    foreach ($names as $name) {
      $result[$name] = [];
      /** @var \Drupal\domain\DomainInterface $domain */
      foreach ($domains as $domain) {
        $domainConfigName = DomainConfigOverrider::getConfigNameByDomain($name, $domain->id());
        // Check domain overridden configuration.
        if ($this->configStorage->exists($domainConfigName)) {
          $result[$name][$domain->id()]['default'] = $this->configStorage->read($domainConfigName);
        }
        foreach ($languages as $language) {
          // Check domain + language overridden configuration.
          $languageConfigName = DomainConfigOverrider::getConfigNameByDomainAndLanguage($name, $domain->id(), $language->getId());
          if ($this->configStorage->exists($languageConfigName)) {
            $result[$name][$domain->id()][$language->getId()] = $this->configStorage->read($languageConfigName);
          }
        }
      }
    }

    return $result;
  }

}
