<?php

declare(strict_types=1);

namespace Drupal\domain_masquerade\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\domain_masquerade\DomainMasqueradeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for domain masquerade switching.
 *
 * Provides routes to activate and clear domain masquerade state, with
 * redirects back to the referring page or front page.
 */
class DomainMasqueradeController extends ControllerBase {

  /**
   * Constructs a DomainMasqueradeController object.
   *
   * @param \Drupal\domain_masquerade\DomainMasqueradeManagerInterface $masqueradeManager
   *   The domain masquerade manager service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   */
  public function __construct(
    protected readonly DomainMasqueradeManagerInterface $masqueradeManager,
    protected readonly RequestStack $requestStack,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('domain_masquerade.manager'),
      $container->get('request_stack')
    );
  }

  /**
   * Switches to masquerading as the specified domain.
   *
   * Validates the domain exists and is active before setting a masquerade state.
   * Redirects to the destination query parameter or front page after switching.
   *
   * @param string $domain_id
   *   The domain ID to masquerade as.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response to the destination or front page.
   */
  public function switchDomain(string $domain_id): RedirectResponse {
    $this->masqueradeManager->setMasqueradedDomain($domain_id);

    return new RedirectResponse($this->getRedirectUrl());
  }

  /**
   * Clears the domain masquerade state.
   *
   * Removes masquerade session data and redirects to the destination query
   * parameter or front page.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response to the destination or front page.
   */
  public function clearMasquerade(): RedirectResponse {
    $this->masqueradeManager->clearMasquerade();

    return new RedirectResponse($this->getRedirectUrl());
  }

  /**
   * Gets a safe redirect URL, avoiding redirect loops.
   *
   * @return string
   *   The URL to redirect to.
   */
  protected function getRedirectUrl(): string {
    $request = $this->requestStack->getCurrentRequest();
    $destination = $request?->query->get('destination');

    if ($destination && !str_contains($destination, '/domain-masquerade/')) {
      return $destination;
    }

    return '/';
  }

}
