<?php

declare(strict_types=1);

namespace Drupal\domain_masquerade;

use Drupal\domain\DomainInterface;
use Drupal\domain\DomainStorageInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Session\SessionInterface;

/**
 * Manages domain masquerade state via session storage.
 *
 * This service handles all operations related to storing and retrieving the
 * domain masquerade state from the user's session.
 */
class DomainMasqueradeManager implements DomainMasqueradeManagerInterface {

  /**
   * Session key for storing the masqueraded domain ID.
   */
  protected const SESSION_KEY = 'domain_masquerade.active_domain_id';

  /**
   * Constructs a DomainMasqueradeManager object.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack service.
   */
  public function __construct(
    protected readonly RequestStack $requestStack,
  ) {}

  /**
   * Gets the domain storage.
   *
   * Lazy-loaded to avoid circular reference during service compilation.
   *
   * @return \Drupal\domain\DomainStorageInterface
   *   The domain storage.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  protected function getDomainStorage(): DomainStorageInterface {
    /** @var \Drupal\domain\DomainStorageInterface $storage */
    $storage = \Drupal::entityTypeManager()->getStorage('domain');
    return $storage;
  }

  /**
   * {@inheritdoc}
   */
  public function getMasqueradedDomain(): ?DomainInterface {
    $domain_id = $this->getSession()?->get(self::SESSION_KEY);

    if ($domain_id === NULL) {
      return NULL;
    }
    $domain = $this->getDomainStorage()->load($domain_id);

    if (!$domain instanceof DomainInterface || !$domain->status()) {
      return NULL;
    }

    return $domain;
  }

  /**
   * {@inheritdoc}
   */
  public function setMasqueradedDomain(string $domain_id): void {
    $domain = $this->getDomainStorage()->load($domain_id);

    if (!$domain instanceof DomainInterface || !$domain->status()) {
      return;
    }

    $this->getSession()?->set(self::SESSION_KEY, $domain_id);
  }

  /**
   * {@inheritdoc}
   */
  public function clearMasquerade(): void {
    $this->getSession()?->remove(self::SESSION_KEY);
  }

  /**
   * {@inheritdoc}
   */
  public function isMasquerading(): bool {
    $domain_id = $this->getSession()?->get(self::SESSION_KEY);
    return $domain_id !== NULL;
  }

  /**
   * Gets the current session.
   *
   * @return \Symfony\Component\HttpFoundation\Session\SessionInterface|null
   *   The session object or NULL if no request is available.
   */
  protected function getSession(): ?SessionInterface {
    return $this->requestStack->getCurrentRequest()?->getSession();
  }

}
