<?php

declare(strict_types=1);

namespace Drupal\domain_masquerade;

use Drupal\domain\DomainInterface;

/**
 * Interface for the Domain Masquerade Manager service.
 *
 * Manages session-based domain masquerading state for multi-domain
 * installations.
 */
interface DomainMasqueradeManagerInterface {

  /**
   * Gets the currently masqueraded domain.
   *
   * Returns the domain object for the masqueraded domain if masquerade is
   * active and the domain is still valid. Returns NULL if no masquerade is
   * active or if the stored domain is no longer valid.
   *
   * @return \Drupal\domain\DomainInterface|null
   *   The masqueraded domain object or NULL.
   */
  public function getMasqueradedDomain(): ?DomainInterface;

  /**
   * Sets the masqueraded domain.
   *
   * Stores the domain ID in the session. Validates that the domain exists
   * and is active before setting. Fails silently if the domain is invalid.
   *
   * @param string $domain_id
   *   The domain ID to masquerade as.
   */
  public function setMasqueradedDomain(string $domain_id): void;

  /**
   * Clears the masquerade state.
   *
   * Removes the masqueraded domain ID from the session.
   */
  public function clearMasquerade(): void;

  /**
   * Checks if masquerade is currently active.
   *
   * @return bool
   *   TRUE if masquerade is active, FALSE otherwise.
   */
  public function isMasquerading(): bool;

}
