<?php

declare(strict_types=1);

namespace Drupal\domain_masquerade;

use Drupal\domain\DomainInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Routing\AdminContext;
use Drupal\domain\DomainNegotiator;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

/**
 * Extends the domain negotiator to support masquerading.
 *
 * This class extends DomainNegotiator (not just implements the interface)
 * because the Domain module's condition plugin type hints the concrete class.
 * It overrides getActiveDomain() to return the masqueraded domain when active.
 * Admin routes always use the real negotiated domain without masquerade override.
 */
class DomainNegotiatorDecorator extends DomainNegotiator {

  /**
   * Flag to prevent infinite recursion during domain negotiation.
   *
   * This can happen because getMasqueradedDomain() loads domain entities,
   * which can trigger domain config resolution, which calls getActiveDomain().
   */
  protected bool $isResolving = FALSE;

  /**
   * Constructs a DomainNegotiatorDecorator object.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack object.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Symfony\Contracts\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   The event dispatcher.
   * @param \Drupal\domain_masquerade\DomainMasqueradeManagerInterface $masqueradeManager
   *   The domain masquerade manager service.
   * @param \Drupal\Core\Routing\AdminContext $adminContext
   *   The admin context service.
   */
  public function __construct(
    RequestStack $requestStack,
    ModuleHandlerInterface $module_handler,
    EntityTypeManagerInterface $entity_type_manager,
    ConfigFactoryInterface $config_factory,
    EventDispatcherInterface $event_dispatcher,
    protected readonly DomainMasqueradeManagerInterface $masqueradeManager,
    protected readonly AdminContext $adminContext,
  ) {
    parent::__construct(
      $requestStack,
      $module_handler,
      $entity_type_manager,
      $config_factory,
      $event_dispatcher
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getActiveDomain($reset = FALSE): ?DomainInterface {
    // Prevent infinite recursion: if we're already resolving, use parent.
    if ($this->isResolving) {
      return parent::getActiveDomain($reset);
    }

    // Admin routes always use the real domain without masquerade override.
    if ($this->adminContext->isAdminRoute()) {
      return parent::getActiveDomain($reset);
    }

    // Set flag to prevent recursion during masquerade domain lookup.
    $this->isResolving = TRUE;
    try {
      $masqueraded_domain = $this->masqueradeManager->getMasqueradedDomain();
      return $masqueraded_domain ?? parent::getActiveDomain($reset);
    }
    finally {
      $this->isResolving = FALSE;
    }
  }

}
