<?php

declare(strict_types=1);

namespace Drupal\domain_masquerade\EventSubscriber;

use Drupal\domain_masquerade\DomainMasqueradeManagerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Event subscriber for domain masquerade validation.
 *
 * Validates that the masqueraded domain still exists and is active on each
 * request. Clears invalid masquerade state silently.
 */
class DomainMasqueradeSubscriber implements EventSubscriberInterface {

  /**
   * Constructs a DomainMasqueradeSubscriber object.
   *
   * @param \Drupal\domain_masquerade\DomainMasqueradeManagerInterface $masqueradeManager
   *   The domain masquerade manager service.
   */
  public function __construct(
    protected DomainMasqueradeManagerInterface $masqueradeManager,
  ) {}

  /**
   * Validates masqueraded domain on each request.
   *
   * Checks if masquerade is active and validates the domain still exists and
   * is active. Clears invalid masquerade state silently without notification.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The request event.
   */
  public function onKernelRequest(RequestEvent $event): void {
    if (!$event->isMainRequest() || !$this->masqueradeManager->isMasquerading()) {
      return;
    }

    if ($this->masqueradeManager->getMasqueradedDomain() === NULL) {
      $this->masqueradeManager->clearMasquerade();
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events[KernelEvents::REQUEST][] = ['onKernelRequest', 100];
    return $events;
  }

}
