<?php

declare(strict_types=1);

namespace Drupal\domain_masquerade\Plugin\Block;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Block\Attribute\Block;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\domain\DomainNegotiatorInterface;
use Drupal\domain\Plugin\Block\DomainBlockBase;
use Drupal\domain_masquerade\DomainMasqueradeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a block for switching between domains via masquerade.
 *
 * This block displays all active domains as clickable links that allow
 * authorized users to masquerade as viewing the site from a different domain.
 */
#[Block(
  id: 'domain_masquerade_switcher',
  admin_label: new TranslatableMarkup('Domain Masquerade Switcher'),
)]
class DomainMasqueradeSwitcherBlock extends DomainBlockBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a DomainMasqueradeSwitcherBlock instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\domain\DomainNegotiatorInterface $domain_negotiator
   *   The domain negotiator service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\domain_masquerade\DomainMasqueradeManagerInterface $masqueradeManager
   *   The domain masquerade manager.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    DomainNegotiatorInterface $domain_negotiator,
    EntityTypeManagerInterface $entity_type_manager,
    AccountInterface $current_user,
    protected DomainMasqueradeManagerInterface $masqueradeManager,
  ) {
    parent::__construct(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $domain_negotiator,
      $entity_type_manager,
      $current_user
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('domain.negotiator'),
      $container->get('entity_type.manager'),
      $container->get('current_user'),
      $container->get('domain_masquerade.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $items = [];
    $active_domain = $this->domainNegotiator->getActiveDomain();

    foreach ($this->domainStorage->loadMultipleSorted() as $domain) {
      if (!$domain->status()) {
        continue;
      }
      $url = Url::fromRoute(
        'domain_masquerade.switch',
        ['domain_id' => $domain->id()],
        ['query' => ['destination' => \Drupal::request()->getRequestUri()]]
      );

      if ($active_domain && $domain->id() === $active_domain->id()) {
        $url->mergeOptions(['attributes' => ['class' => ['is-active']]]);
      }

      $items[] = Link::fromTextAndUrl($domain->label(), $url);
    }

    if ($this->masqueradeManager->isMasquerading()) {
      $clear_url = Url::fromRoute('domain_masquerade.clear', [], [
        'query' => ['destination' => \Drupal::request()->getRequestUri()],
      ]);
      $items[] = Link::fromTextAndUrl($this->t('Clear masquerade'), $clear_url);
    }

    return [
      '#theme' => 'item_list',
      '#items' => $items,
      '#attached' => [
        'library' => [
          'domain_masquerade/switcher_block',
        ],
      ],
      '#cache' => [
        'contexts' => $this->getCacheContexts(),
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function access(
    AccountInterface $account,
    $return_as_object = FALSE
  ): AccessResult|bool {
    $access = AccessResult::allowedIfHasPermission(
      $account,
      'masquerade as domain'
    );
    return $return_as_object ? $access : $access->isAllowed();
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts(): array {
    return ['user', 'url.path'];
  }

}
