<?php

declare(strict_types=1);

namespace Drupal\Tests\domain_masquerade\Kernel;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;

/**
 * Tests for the DomainMasqueradeController.
 *
 * @group domain_masquerade
 */
class DomainMasqueradeControllerTest extends DomainMasqueradeKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install routing configuration.
    $this->container->get('router.builder')->rebuild();
  }

  /**
   * Tests switch route sets masquerade and redirects.
   */
  public function testSwitchRouteSetsМasqueradeAndRedirects(): void {
    $manager = $this->container->get('domain_masquerade.manager');

    // Verify no masquerade is active initially.
    $this->assertFalse($manager->isMasquerading());

    // Create request with session.
    $request = Request::create('/domain-masquerade/switch/secondary_domain');
    $request->setSession($this->container->get('session'));
    $this->container->get('request_stack')->push($request);

    // Get controller and call switchDomain.
    $controller = $this->container->get('domain_masquerade.controller');
    $response = $controller->switchDomain('secondary_domain');

    // Verify redirect response.
    $this->assertEquals(302, $response->getStatusCode());

    // Verify masquerade was set.
    $this->assertTrue($manager->isMasquerading());
    $masqueraded = $manager->getMasqueradedDomain();
    $this->assertEquals('secondary_domain', $masqueraded->id());
  }

  /**
   * Tests clear route clears masquerade and redirects.
   */
  public function testClearRouteClearsMasqueradeAndRedirects(): void {
    $manager = $this->container->get('domain_masquerade.manager');

    // Set masquerade state.
    $request = Request::create('/');
    $request->setSession($this->container->get('session'));
    $this->container->get('request_stack')->push($request);

    $manager->setMasqueradedDomain('secondary_domain');
    $this->assertTrue($manager->isMasquerading());

    // Call clearMasquerade controller method.
    $controller = $this->container->get('domain_masquerade.controller');
    $response = $controller->clearMasquerade();

    // Verify redirect response.
    $this->assertEquals(302, $response->getStatusCode());

    // Verify masquerade was cleared.
    $this->assertFalse($manager->isMasquerading());
  }

  /**
   * Tests access denied for users without permission.
   */
  public function testAccessDeniedForUnauthorizedUsers(): void {
    // Verify the route requires the correct permission.
    $route_provider = $this->container->get('router.route_provider');
    $route = $route_provider->getRouteByName('domain_masquerade.switch');

    $requirements = $route->getRequirements();
    $this->assertEquals('masquerade as domain', $requirements['_permission']);

    // Also test clear route.
    $clear_route = $route_provider->getRouteByName('domain_masquerade.clear');
    $clear_requirements = $clear_route->getRequirements();
    $this->assertEquals('masquerade as domain', $clear_requirements['_permission']);
  }

  /**
   * Tests event subscriber clears invalid domain on request.
   */
  public function testEventSubscriberClearsInvalidDomain(): void {
    $manager = $this->container->get('domain_masquerade.manager');

    // Create request with session.
    $request = Request::create('/');
    $request->setSession($this->container->get('session'));
    $this->container->get('request_stack')->push($request);

    // Set masquerade to inactive domain (should be invalid).
    $session = $request->getSession();
    $session->set('domain_masquerade.active_domain_id', 'inactive_domain');

    // Verify masquerade returns NULL for inactive domain.
    $this->assertNull($manager->getMasqueradedDomain());

    // Now delete a domain to test cleanup.
    $manager->setMasqueradedDomain('secondary_domain');
    $this->assertTrue($manager->isMasquerading());

    // Delete the masqueraded domain.
    $this->domains['secondary']->delete();

    // Simulate request event by calling the subscriber directly.
    $subscriber = $this->container->get('domain_masquerade.event_subscriber');
    $kernel = $this->container->get('http_kernel');

    $event = new \Symfony\Component\HttpKernel\Event\RequestEvent(
      $kernel,
      $request,
      \Symfony\Component\HttpKernel\HttpKernelInterface::MAIN_REQUEST
    );

    $subscriber->onKernelRequest($event);

    // Verify masquerade was cleared.
    $this->assertFalse($manager->isMasquerading());
  }

}
