<?php

declare(strict_types=1);

namespace Drupal\Tests\domain_masquerade\Unit;

use Drupal\domain\DomainInterface;
use Drupal\domain\DomainStorageInterface;
use Drupal\domain_masquerade\DomainMasqueradeManager;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Session\SessionInterface;

/**
 * Testable subclass that allows injecting domain storage.
 */
class TestableDomainMasqueradeManager extends DomainMasqueradeManager {

  /**
   * The domain storage for testing.
   */
  protected DomainStorageInterface $testDomainStorage;

  /**
   * Sets the domain storage for testing.
   */
  public function setDomainStorage(DomainStorageInterface $storage): void {
    $this->testDomainStorage = $storage;
  }

  /**
   * {@inheritdoc}
   */
  protected function getDomainStorage(): DomainStorageInterface {
    return $this->testDomainStorage;
  }

}

/**
 * Unit tests for DomainMasqueradeManager service.
 *
 * @group domain_masquerade
 * @coversDefaultClass \Drupal\domain_masquerade\DomainMasqueradeManager
 */
class DomainMasqueradeManagerTest extends UnitTestCase {

  /**
   * The domain masquerade manager service.
   *
   * @var \Drupal\Tests\domain_masquerade\Unit\TestableDomainMasqueradeManager
   */
  protected $manager;

  /**
   * Mock session.
   *
   * @var \Symfony\Component\HttpFoundation\Session\SessionInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $session;

  /**
   * Mock domain storage.
   *
   * @var \Drupal\domain\DomainStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $domainStorage;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mock session.
    $this->session = $this->createMock(SessionInterface::class);

    // Create mock request.
    $request = $this->createMock(Request::class);
    $request->method('getSession')
      ->willReturn($this->session);

    // Create mock request stack.
    $request_stack = $this->createMock(RequestStack::class);
    $request_stack->method('getCurrentRequest')
      ->willReturn($request);

    // Create mock domain storage.
    $this->domainStorage = $this->createMock(DomainStorageInterface::class);

    // Create the testable manager service.
    $this->manager = new TestableDomainMasqueradeManager($request_stack);
    $this->manager->setDomainStorage($this->domainStorage);
  }

  /**
   * Tests getMasqueradedDomain returns NULL when no masquerade active.
   *
   * @covers ::getMasqueradedDomain
   */
  public function testGetMasqueradedDomainReturnsNullWhenNoMasqueradeActive(): void {
    $this->session->expects($this->once())
      ->method('get')
      ->with('domain_masquerade.active_domain_id')
      ->willReturn(NULL);

    $result = $this->manager->getMasqueradedDomain();
    $this->assertNull($result);
  }

  /**
   * Tests setMasqueradedDomain stores domain ID in session.
   *
   * @covers ::setMasqueradedDomain
   */
  public function testSetMasqueradedDomainStoresDomainIdInSession(): void {
    $domain_id = 'test_domain';

    // Create mock domain.
    $domain = $this->createMock(DomainInterface::class);
    $domain->method('status')
      ->willReturn(TRUE);

    $this->domainStorage->expects($this->once())
      ->method('load')
      ->with($domain_id)
      ->willReturn($domain);

    $this->session->expects($this->once())
      ->method('set')
      ->with('domain_masquerade.active_domain_id', $domain_id);

    $this->manager->setMasqueradedDomain($domain_id);
  }

  /**
   * Tests clearMasquerade removes session data.
   *
   * @covers ::clearMasquerade
   */
  public function testClearMasqueradeRemovesSessionData(): void {
    $this->session->expects($this->once())
      ->method('remove')
      ->with('domain_masquerade.active_domain_id');

    $this->manager->clearMasquerade();
  }

  /**
   * Tests isMasquerading returns correct boolean state.
   *
   * @covers ::isMasquerading
   */
  public function testIsMasqueradingReturnsCorrectBooleanState(): void {
    // Test when no masquerade is active.
    $this->session->expects($this->once())
      ->method('get')
      ->with('domain_masquerade.active_domain_id')
      ->willReturn(NULL);

    $result = $this->manager->isMasquerading();
    $this->assertFalse($result);

    // Reset the manager for second test.
    $this->setUp();

    // Test when masquerade is active.
    $this->session->expects($this->once())
      ->method('get')
      ->with('domain_masquerade.active_domain_id')
      ->willReturn('test_domain');

    $result = $this->manager->isMasquerading();
    $this->assertTrue($result);
  }

}
