<?php

declare(strict_types=1);

namespace Drupal\Tests\domain_masquerade\Unit;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Routing\AdminContext;
use Drupal\domain\DomainInterface;
use Drupal\domain\DomainStorageInterface;
use Drupal\domain_masquerade\DomainMasqueradeManagerInterface;
use Drupal\domain_masquerade\DomainNegotiatorDecorator;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

/**
 * Tests the DomainNegotiatorDecorator service.
 *
 * @coversDefaultClass \Drupal\domain_masquerade\DomainNegotiatorDecorator
 * @group domain_masquerade
 */
class DomainNegotiatorDecoratorTest extends UnitTestCase {

  /**
   * The domain masquerade manager.
   *
   * @var \Drupal\domain_masquerade\DomainMasqueradeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $masqueradeManager;

  /**
   * The admin context service.
   *
   * @var \Drupal\Core\Routing\AdminContext|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $adminContext;

  /**
   * The decorator under test.
   *
   * @var \Drupal\domain_masquerade\DomainNegotiatorDecorator
   */
  protected $decorator;

  /**
   * The mock domain storage.
   *
   * @var \Drupal\domain\DomainStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $domainStorage;

  /**
   * A mock real domain.
   *
   * @var \Drupal\domain\DomainInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $realDomain;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mock dependencies for parent DomainNegotiator.
    $request = $this->createMock(Request::class);
    $request->method('getHttpHost')->willReturn('example.com');

    $requestStack = $this->createMock(RequestStack::class);
    $requestStack->method('getCurrentRequest')->willReturn($request);

    $moduleHandler = $this->createMock(ModuleHandlerInterface::class);

    // Create a real domain mock.
    $this->realDomain = $this->createMock(DomainInterface::class);
    $this->realDomain->method('id')->willReturn('real_domain');

    // Setup domain storage to return the real domain.
    $this->domainStorage = $this->createMock(DomainStorageInterface::class);
    $this->domainStorage->method('loadByHostname')->willReturn($this->realDomain);
    $this->domainStorage->method('prepareHostname')->willReturnArgument(0);

    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')
      ->with('domain')
      ->willReturn($this->domainStorage);

    $configFactory = $this->createMock(ConfigFactoryInterface::class);
    $eventDispatcher = $this->createMock(EventDispatcherInterface::class);

    // Create our service mocks.
    $this->masqueradeManager = $this->createMock(DomainMasqueradeManagerInterface::class);
    $this->adminContext = $this->createMock(AdminContext::class);

    $this->decorator = new DomainNegotiatorDecorator(
      $requestStack,
      $moduleHandler,
      $entityTypeManager,
      $configFactory,
      $eventDispatcher,
      $this->masqueradeManager,
      $this->adminContext
    );
  }

  /**
   * Tests getActiveDomain() returns masqueraded domain when active.
   *
   * @covers ::getActiveDomain
   */
  public function testGetActiveDomainReturnsMasqueradedDomain(): void {
    $masqueraded_domain = $this->createMock(DomainInterface::class);
    $masqueraded_domain->method('id')->willReturn('masqueraded_domain');

    // Not an admin route.
    $this->adminContext->expects($this->once())
      ->method('isAdminRoute')
      ->willReturn(FALSE);

    // Masquerade is active.
    $this->masqueradeManager->expects($this->once())
      ->method('getMasqueradedDomain')
      ->willReturn($masqueraded_domain);

    $result = $this->decorator->getActiveDomain();

    $this->assertSame($masqueraded_domain, $result);
  }

  /**
   * Tests getActiveDomain() uses parent negotiation when no masquerade.
   *
   * @covers ::getActiveDomain
   */
  public function testGetActiveDomainUsesParentWhenNoMasquerade(): void {
    // Not an admin route.
    $this->adminContext->expects($this->once())
      ->method('isAdminRoute')
      ->willReturn(FALSE);

    // No masquerade active.
    $this->masqueradeManager->expects($this->once())
      ->method('getMasqueradedDomain')
      ->willReturn(NULL);

    $result = $this->decorator->getActiveDomain();

    // Should return the real domain from parent negotiation.
    $this->assertSame($this->realDomain, $result);
  }

  /**
   * Tests admin routes always use real domain (no masquerade override).
   *
   * @covers ::getActiveDomain
   */
  public function testAdminRoutesAlwaysUseRealDomain(): void {
    // Admin route detected.
    $this->adminContext->expects($this->once())
      ->method('isAdminRoute')
      ->willReturn(TRUE);

    // Masquerade manager should not be called.
    $this->masqueradeManager->expects($this->never())
      ->method('getMasqueradedDomain');

    $result = $this->decorator->getActiveDomain();

    // Should return the real domain from parent negotiation.
    $this->assertSame($this->realDomain, $result);
  }

  /**
   * Tests that parent class methods are still accessible.
   *
   * @covers ::negotiateActiveHostname
   * @covers ::setHttpHost
   * @covers ::getHttpHost
   */
  public function testParentMethodsAccessible(): void {
    // Test that we can call parent methods.
    $hostname = $this->decorator->negotiateActiveHostname();
    $this->assertEquals('example.com', $hostname);

    // Test setHttpHost and getHttpHost.
    $this->decorator->setHttpHost('test.example.com');
    $this->assertEquals('test.example.com', $this->decorator->getHttpHost());
  }

}
