<?php

declare(strict_types=1);

namespace Drupal\Tests\domain_masquerade\Unit\Plugin\Block;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Cache\Context\CacheContextsManager;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Session\AccountInterface;
use Drupal\domain\DomainInterface;
use Drupal\domain\DomainNegotiatorInterface;
use Drupal\domain\DomainStorageInterface;
use Drupal\domain_masquerade\DomainMasqueradeManagerInterface;
use Drupal\domain_masquerade\Plugin\Block\DomainMasqueradeSwitcherBlock;
use Drupal\Tests\UnitTestCase;

/**
 * Tests the Domain Masquerade Switcher Block plugin.
 *
 * @group domain_masquerade
 * @coversDefaultClass \Drupal\domain_masquerade\Plugin\Block\DomainMasqueradeSwitcherBlock
 */
class DomainMasqueradeSwitcherBlockTest extends UnitTestCase {

  /**
   * The mocked domain negotiator.
   *
   * @var \Drupal\domain\DomainNegotiatorInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $domainNegotiator;

  /**
   * The mocked domain storage.
   *
   * @var \Drupal\domain\DomainStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $domainStorage;

  /**
   * The mocked current user.
   *
   * @var \Drupal\Core\Session\AccountInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $currentUser;

  /**
   * The mocked domain masquerade manager.
   *
   * @var \Drupal\domain_masquerade\DomainMasqueradeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $masqueradeManager;

  /**
   * The block plugin under test.
   *
   * @var \Drupal\domain_masquerade\Plugin\Block\DomainMasqueradeSwitcherBlock
   */
  protected $block;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->domainNegotiator = $this->createMock(DomainNegotiatorInterface::class);
    $this->domainStorage = $this->createMock(DomainStorageInterface::class);
    $this->currentUser = $this->createMock(AccountInterface::class);
    $this->masqueradeManager = $this->createMock(DomainMasqueradeManagerInterface::class);

    $configuration = [];
    $plugin_id = 'domain_masquerade_switcher';
    $plugin_definition = [
      'provider' => 'domain_masquerade',
    ];

    // Create a partial mock that doesn't call the parent constructor.
    $this->block = $this->getMockBuilder(DomainMasqueradeSwitcherBlock::class)
      ->disableOriginalConstructor()
      ->onlyMethods([])
      ->getMock();

    // Manually inject dependencies via reflection.
    $reflection = new \ReflectionClass($this->block);

    $negotiatorProperty = $reflection->getProperty('domainNegotiator');
    $negotiatorProperty->setAccessible(TRUE);
    $negotiatorProperty->setValue($this->block, $this->domainNegotiator);

    $storageProperty = $reflection->getProperty('domainStorage');
    $storageProperty->setAccessible(TRUE);
    $storageProperty->setValue($this->block, $this->domainStorage);

    $userProperty = $reflection->getProperty('currentUser');
    $userProperty->setAccessible(TRUE);
    $userProperty->setValue($this->block, $this->currentUser);

    $managerProperty = $reflection->getProperty('masqueradeManager');
    $managerProperty->setAccessible(TRUE);
    $managerProperty->setValue($this->block, $this->masqueradeManager);

    // Mock the string translation.
    $stringTranslation = $this->getStringTranslationStub();
    $this->block->setStringTranslation($stringTranslation);

    // Set up container for AccessResult and request handling.
    $cache_contexts_manager = $this->createMock(CacheContextsManager::class);
    $cache_contexts_manager->method('assertValidTokens')->willReturn(TRUE);

    // Mock the request stack and request for \Drupal::request() calls.
    $request = $this->getMockBuilder('Symfony\Component\HttpFoundation\Request')
      ->disableOriginalConstructor()
      ->getMock();
    $request->method('getRequestUri')->willReturn('/test-path');

    $request_stack = $this->getMockBuilder('Symfony\Component\HttpFoundation\RequestStack')
      ->disableOriginalConstructor()
      ->getMock();
    $request_stack->method('getCurrentRequest')->willReturn($request);

    $container = new ContainerBuilder();
    $container->set('cache_contexts_manager', $cache_contexts_manager);
    $container->set('request_stack', $request_stack);
    \Drupal::setContainer($container);
  }

  /**
   * Tests that block build renders list of domains.
   *
   * @covers ::build
   */
  public function testBlockBuildRendersListOfDomains(): void {
    $domain1 = $this->createMock(DomainInterface::class);
    $domain1->method('id')->willReturn('domain1');
    $domain1->method('label')->willReturn('Domain 1');
    $domain1->method('status')->willReturn(TRUE);
    $domain1->method('isActive')->willReturn(TRUE);

    $domain2 = $this->createMock(DomainInterface::class);
    $domain2->method('id')->willReturn('domain2');
    $domain2->method('label')->willReturn('Domain 2');
    $domain2->method('status')->willReturn(TRUE);
    $domain2->method('isActive')->willReturn(FALSE);

    $this->domainStorage
      ->expects($this->once())
      ->method('loadMultipleSorted')
      ->willReturn(['domain1' => $domain1, 'domain2' => $domain2]);

    $this->masqueradeManager
      ->method('isMasquerading')
      ->willReturn(FALSE);

    $this->domainNegotiator
      ->method('getActiveDomain')
      ->willReturn($domain1);

    $build = $this->block->build();

    $this->assertIsArray($build);
    $this->assertEquals('item_list', $build['#theme']);
    $this->assertArrayHasKey('#items', $build);
    $this->assertCount(2, $build['#items']);
  }

  /**
   * Tests active/masqueraded domain has is-active class.
   *
   * @covers ::build
   */
  public function testActiveMasqueradedDomainHasIsActiveClass(): void {
    $domain1 = $this->createMock(DomainInterface::class);
    $domain1->method('id')->willReturn('domain1');
    $domain1->method('label')->willReturn('Domain 1');
    $domain1->method('status')->willReturn(TRUE);
    $domain1->method('isActive')->willReturn(FALSE);

    $domain2 = $this->createMock(DomainInterface::class);
    $domain2->method('id')->willReturn('domain2');
    $domain2->method('label')->willReturn('Domain 2');
    $domain2->method('status')->willReturn(TRUE);
    $domain2->method('isActive')->willReturn(FALSE);

    $this->domainStorage
      ->method('loadMultipleSorted')
      ->willReturn(['domain1' => $domain1, 'domain2' => $domain2]);

    // Masquerading as domain2.
    $this->masqueradeManager
      ->method('isMasquerading')
      ->willReturn(TRUE);

    $this->masqueradeManager
      ->method('getMasqueradedDomain')
      ->willReturn($domain2);

    $this->domainNegotiator
      ->method('getActiveDomain')
      ->willReturn($domain2);

    $build = $this->block->build();

    $this->assertIsArray($build);
    $this->assertArrayHasKey('#items', $build);

    // Find the domain2 item and check for is-active class.
    $found_active = FALSE;
    foreach ($build['#items'] as $item) {
      // Items are Link objects.
      if ($item instanceof \Drupal\Core\Link) {
        $url = $item->getUrl();
        $text = $item->getText();
        if (strpos((string) $text, 'Domain 2') !== FALSE) {
          $attributes = $url->getOption('attributes');
          if (isset($attributes['class']) && in_array('is-active', $attributes['class'])) {
            $found_active = TRUE;
            break;
          }
        }
      }
    }

    $this->assertTrue($found_active, 'Active/masqueraded domain should have is-active class');
  }

  /**
   * Tests "Clear masquerade" link appears when masquerade active.
   *
   * @covers ::build
   */
  public function testClearMasqueradeLinkAppearsWhenMasqueradeActive(): void {
    $domain1 = $this->createMock(DomainInterface::class);
    $domain1->method('id')->willReturn('domain1');
    $domain1->method('label')->willReturn('Domain 1');
    $domain1->method('status')->willReturn(TRUE);

    $this->domainStorage
      ->method('loadMultipleSorted')
      ->willReturn(['domain1' => $domain1]);

    // Masquerade is active.
    $this->masqueradeManager
      ->method('isMasquerading')
      ->willReturn(TRUE);

    $this->masqueradeManager
      ->method('getMasqueradedDomain')
      ->willReturn($domain1);

    $this->domainNegotiator
      ->method('getActiveDomain')
      ->willReturn($domain1);

    $build = $this->block->build();

    $this->assertIsArray($build);
    $this->assertArrayHasKey('#items', $build);

    // Check that the clear link is present.
    $found_clear_link = FALSE;
    foreach ($build['#items'] as $item) {
      // Items are Link objects.
      if ($item instanceof \Drupal\Core\Link) {
        $text = $item->getText();
        if (strpos((string) $text, 'Clear') !== FALSE) {
          $found_clear_link = TRUE;
          break;
        }
      }
    }

    $this->assertTrue($found_clear_link, 'Clear masquerade link should appear when masquerade is active');
  }

  /**
   * Tests access check enforces permission.
   *
   * @covers ::access
   */
  public function testAccessCheckEnforcesPermission(): void {
    $account = $this->createMock(AccountInterface::class);

    // Test with permission.
    $account->expects($this->once())
      ->method('hasPermission')
      ->with('masquerade as domain')
      ->willReturn(TRUE);

    $result = $this->block->access($account, TRUE);
    $this->assertInstanceOf(AccessResult::class, $result);
    $this->assertTrue($result->isAllowed());

    // Create new block instance for second test.
    $this->setUp();
    $account2 = $this->createMock(AccountInterface::class);

    // Test without permission.
    $account2->expects($this->once())
      ->method('hasPermission')
      ->with('masquerade as domain')
      ->willReturn(FALSE);

    $result2 = $this->block->access($account2, TRUE);
    $this->assertInstanceOf(AccessResult::class, $result2);
    $this->assertFalse($result2->isAllowed());
  }

}
