<?php

declare(strict_types=1);

namespace Drupal\Tests\domain_menu_links\Functional;

use Drupal\domain_menu_links\DomainMenuLinksConstants;

/**
 * Tests the toolbar domain menu.
 *
 * @group domain_menu_links
 */
class DomainMenuLinksMenuTest extends DomainMenuLinksTestBase {

  /**
   * Path used for tests.
   */
  const TEST_PATH = 'node/';

  /**
   * Toolbar menu xpath.
   */
  const TOOLBAR_XPATH = "//div[@class='toolbar-menu-administration']/ul[contains(@class, 'toolbar-menu')]";

  /**
   * Toolbar domain menu links xpath.
   */
  private string $domainMenuLinksXpath = "";

  /**
   * Toolbar domain menu links list.
   */
  private array $toolbarDomainMenuLinks = [];

  /**
   * Test setup.
   */
  protected function setUp(): void {
    parent::setUp();

    $this->addModules([
      'toolbar',
      'admin_toolbar',
    ]);
    $this->domainMenuLinksXpath = self::TOOLBAR_XPATH . "//a[contains(@class, '" . DomainMenuLinksConstants::DOMAIN_MENU_LINK_CLASS . "')]";
  }

  /**
   * Test parent menu link.
   */
  public function testDomainMenuParentLink(): void {
    $xpath = self::TOOLBAR_XPATH . "/li/a[contains(@class, 'toolbar-icon-domain-menu')]";

    // The domain menu should not exist if there are no domains.
    $editor = $this->createTestUser([DomainMenuLinksConstants::DOMAIN_MENU_PERMISSION]);
    $this->drupalLogin($editor);
    $this->drupalGet(self::TEST_PATH);
    $this->assertEmpty($this->xpath($xpath), 'Domain menu exists when there are no domains.');

    $this->domainCreateTestDomains(5);

    // A user without permission should not see the domain menu.
    $author = $this->createTestUser();
    $this->drupalLogin($author);
    $this->drupalGet(self::TEST_PATH);
    $this->assertEmpty($this->xpath($xpath), 'User without permission can see the domain menu.');

    // A user with permission should see the domain menu.
    $this->drupalLogin($editor);
    $this->drupalGet(self::TEST_PATH);
    $this->assertNotEmpty($this->xpath($xpath), 'User with permission cannot see the domain menu.');
  }

  /**
   * Test dropdown links.
   */
  public function testDomainMenuDropdownLinks(): void {
    $this->domainCreateTestDomains(5);

    // A user without permission should not see the domain menu links.
    $editor = $this->createTestUser();
    $this->drupalLogin($editor);
    $this->drupalGet(self::TEST_PATH);
    $this->assertEmpty($this->xpath($this->domainMenuLinksXpath), 'User without permission can see the domain menu links.');

    // A user with permission should see the domain menu links.
    $admin = $this->createTestUser([DomainMenuLinksConstants::DOMAIN_MENU_PERMISSION]);
    $this->drupalLogin($admin);
    $this->drupalGet(self::TEST_PATH);
    $this->assertNotEmpty($this->xpath($this->domainMenuLinksXpath), 'User with permission cannot see the domain menu links.');

    // Test if links are sorted by domain weights.
    $this->refreshPage();
    $this->testMenuLinksOrder();

    // Invert domain weights for the next test.
    $domains_sorted = $this->getDomainsSorted();
    $domains_length = count($domains_sorted);

    foreach ($domains_sorted as $domain) {
      $domain->set('weight', $domains_length);
      $domain->save();
      $domains_length--;
    }

    // Test menu links order after domain updates.
    $this->refreshPage();
    $this->testMenuLinksOrder();

    // Remove some domains for the next test.
    $loop_index = 0;

    foreach ($this->getDomainsSorted() as $domain) {
      if ($loop_index === 2 || $loop_index === 4) {
        $domain->delete();
      }

      $loop_index++;
    }

    // Test menu links after deleting some domains.
    $this->refreshPage();
    $this->testMenuLinksOrder();
  }

  /**
   * Creates a user for tests.
   *
   * @param array $permissions
   *   The user permissions.
   *
   * @return \Drupal\user\Entity\User|false
   *   A fully loaded user object or FALSE if account creation fails.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function createTestUser(array $permissions = []): object|bool {
    $permissions[] = 'access toolbar';
    return $this->drupalCreateUser($permissions);
  }

  /**
   * Refresh the toolbar domain menu links list.
   */
  protected function refreshPage(): void {
    $this->drupalGet(self::TEST_PATH);
    $this->toolbarDomainMenuLinks = $this->xpath($this->domainMenuLinksXpath);
  }

  /**
   * Test if domains and the menu links have the same weight.
   */
  protected function testMenuLinksOrder(): void {
    foreach ($this->getDomainsSorted() as $domain) {
      $menu_link_label = current($this->toolbarDomainMenuLinks)->getText();
      $this->assertEquals($domain->label(), $menu_link_label, 'Domain and menu link are not in the same order.');
      next($this->toolbarDomainMenuLinks);
    }
  }

}
