<?php

namespace Drupal\domain_menu_links\Plugin\Derivative;

use Drupal\Component\Plugin\Derivative\DeriverBase;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\domain_menu_links\DomainMenuLinksConstants;
use Drupal\domain_menu_links\Plugin\Menu\DomainMenuLink;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Derivative class that provides domains menu links.
 */
class DomainMenuDeriver extends DeriverBase implements ContainerDeriverInterface {

  /**
   * Class constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $base_plugin_id): object {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinitions($base_plugin_definition): array {
    $derivatives = [];
    $domains = $this->entityTypeManager->getStorage('domain')->loadMultiple();

    if (empty($domains)) {
      return $derivatives;
    }

    $derivatives[] = $base_plugin_definition;

    /** @var \Drupal\domain\Entity\Domain[] $domains */
    foreach ($domains as $id => $domain) {
      if (!$domain->status()) {
        continue;
      }

      $derivatives[$id] = [
        'title' => $domain->label(),
        'url' => $domain->getRawPath(),
        'weight' => $domain->getWeight(),
        'parent' => 'domain_menu_links.domain',
        'provider' => 'domain_menu_links',
        'class' => DomainMenuLink::class,
        'options' => [
          'attributes' => [
            'target' => '_blank',
            'class' => [
              DomainMenuLinksConstants::DOMAIN_MENU_LINK_CLASS,
            ],
          ],
        ],
        'cache' => [
          'tags' => $domain->getCacheTags(),
          'context' => $domain->getCacheContexts(),
          'max-age' => $domain->getCacheMaxAge(),
        ],
      ];
    }

    return $derivatives;
  }

}
