<?php

namespace Drupal\domain_menu_links\Plugin\Menu;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Menu\MenuLinkDefault;
use Drupal\Core\Menu\StaticMenuLinkOverridesInterface;
use Drupal\domain\DomainInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a custom menu link for domains.
 *
 * @Menu(
 *   id = "domain_menu_links.domain",
 *   deriver = "Drupal\Core\Menu\MenuLinkDefault"
 * )
 */
class DomainMenuLink extends MenuLinkDefault {

  /**
   * The domain entity.
   */
  protected ?DomainInterface $domain = NULL;

  /**
   * Constructs a new DomainMenuLink.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Menu\StaticMenuLinkOverridesInterface $staticOverride
   *   The static override storage.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity Type Manager Service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    StaticMenuLinkOverridesInterface $staticOverride,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $staticOverride);

    $menu_link_id = explode(':', $this->pluginDefinition['id']);

    if (isset($menu_link_id[1])) {
      if ($domain = $this->entityTypeManager->getStorage('domain')->load($menu_link_id[1])) {
        /** @var \Drupal\domain\DomainInterface $domain */
        $this->domain = $domain;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('menu_link.static.overrides'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts(): array {
    return $this->domain ? $this->domain->getCacheContexts() : [];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags(): array {
    return $this->domain ? $this->domain->getCacheTags() : [];
  }

}
