<?php

namespace Drupal\domain_menu_links\Plugin\Menu;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Menu\MenuLinkDefault;
use Drupal\Core\Menu\StaticMenuLinkOverridesInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\domain_menu_links\DomainMenuLinksConstants;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Domain menu parent link class.
 *
 * @Menu(
 *   id = "domain_menu_links.domain",
 *   deriver = "Drupal\Core\Menu\MenuLinkDefault"
 * )
 */
class DomainMenu extends MenuLinkDefault {

  /**
   * Constructs a new DomainMenu.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $pluginId
   *   The plugin ID for the plugin instance.
   * @param mixed $pluginDefinition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Menu\StaticMenuLinkOverridesInterface $staticOverride
   *   The static override storage.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity Type Manager Service.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user account proxy.
   */
  public function __construct(
    array $configuration,
    $pluginId,
    $pluginDefinition,
    StaticMenuLinkOverridesInterface $staticOverride,
    protected ConfigFactoryInterface $configFactory,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected AccountProxyInterface $currentUser,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition, $staticOverride);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('menu_link.static.overrides'),
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('current_user'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getWeight() {
    $config = $this->configFactory->get(DomainMenuLinksConstants::SETTINGS);

    return $config->get(DomainMenuLinksConstants::MENU_WEIGHT_FIELD);
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts(): array {
    return ['url.site'];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags(): array {
    return ['domain_list'];
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled(): bool {
    $domains = $this->entityTypeManager->getStorage('domain')->loadMultiple();

    return $this->currentUser->hasPermission(DomainMenuLinksConstants::DOMAIN_MENU_PERMISSION) && !empty($domains);
  }

}
