<?php

declare(strict_types=1);

namespace Drupal\Tests\domain_menu_links\Functional;

use Drupal\domain_menu_links\DomainMenuLinksConstants;
use Drupal\user\UserInterface;

/**
 * Tests the toolbar domain menu.
 *
 * @group domain_menu_links
 */
class DomainMenuLinksMenuTest extends DomainMenuLinksTestBase {

  /**
   * Path used for tests.
   */
  const TEST_PATH = 'node/';

  /**
   * Toolbar menu xpath.
   */
  const TOOLBAR_XPATH = "//div[@class='toolbar-menu-administration']/ul[contains(@class, 'toolbar-menu')]";

  /**
   * A user with access to the domain menu.
   */
  protected UserInterface|false $allowedUser;

  /**
   * Toolbar domain menu xpath.
   */
  protected string $domainMenuXpath = "";

  /**
   * Toolbar domain menu links xpath.
   */
  protected string $domainMenuLinksXpath = "";

  /**
   * Toolbar domain menu links list.
   */
  protected array $toolbarDomainMenuLinks = [];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->allowedUser = $this->createTestUser([DomainMenuLinksConstants::DOMAIN_MENU_PERMISSION]);
    // @todo use xpath builder.
    $this->domainMenuXpath = self::TOOLBAR_XPATH . "/li/a[contains(@class, 'toolbar-icon-domain-menu')]";
    $this->domainMenuLinksXpath = self::TOOLBAR_XPATH . "//a[contains(@class, '" . DomainMenuLinksConstants::DOMAIN_MENU_LINK_CLASS . "')]";
  }

  /**
   * Test domain menu parent link.
   *
   * @covers \Drupal\domain_menu_links\Plugin\Menu\DomainMenu
   */
  public function testDomainMenuParentLink(): void {
    // The domain menu should not exist if there are no domains.
    $this->drupalLogin($this->allowedUser);
    $this->drupalGet(self::TEST_PATH);
    $this->assertSession()->elementNotExists('xpath', $this->domainMenuXpath);

    $this->domainCreateTestDomains(5);

    // A user with permission should see the domain menu.
    $this->drupalGet(self::TEST_PATH);
    $parent_link = $this->xpath($this->domainMenuXpath);
    $this->assertCount(1, $parent_link, 'There must be a single parent link. Getting ' . count($parent_link));

    // Test if menu label is correct.
    $this->assertEquals('Domains', current($parent_link)->getText());

    // A user without permission should not see the domain menu.
    $non_allowed_user = $this->createTestUser();
    $this->drupalLogin($non_allowed_user);
    $this->drupalGet(self::TEST_PATH);
    $this->assertSession()->elementNotExists('xpath', $this->domainMenuXpath);
  }

  /**
   * Test dropdown links.
   *
   * @covers \Drupal\domain_menu_links\Plugin\Derivative\DomainMenuDeriver
   */
  public function testDomainMenuDropdownLinks(): void {
    $this->domainCreateTestDomains(10);

    // A user with permission should see the domain menu links.
    $this->drupalLogin($this->allowedUser);
    $this->drupalGet(self::TEST_PATH);
    $this->assertSession()->elementExists('xpath', $this->domainMenuLinksXpath);
    $this->refreshPage();
    $this->testMenuLinksSizeAndOrder();

    // Invert domain weights for the next test.
    $domains = $this->getDomainsSorted();
    $domains_length = count($domains);

    foreach ($domains as $domain) {
      /** @var \Drupal\domain\Entity\Domain $domain */
      $domain->set('weight', $domains_length);
      $domain->save();
      $domains_length--;
    }

    // Test menu links order after domain updates.
    $this->refreshPage();
    $this->testMenuLinksSizeAndOrder();

    // Disable some domains for the next test.
    foreach ($domains as $key => $domain) {
      if ($key % 2 === 0) {
        $domain->disable();
      }
    }

    $this->refreshPage();
    $this->testMenuLinksSizeAndOrder();

    // Remove some domains for the next test.
    foreach ($domains as $key => $domain) {
      $key % 2 === 0 ? $domain->enable() : $domain->delete();
    }

    $this->refreshPage();
    $this->testMenuLinksSizeAndOrder();
  }

  /**
   * Creates a user for tests.
   *
   * @param array $permissions
   *   The user permissions.
   *
   * @return \Drupal\user\Entity\User|false
   *   A fully loaded user object or FALSE if account creation fails.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function createTestUser(array $permissions = []): object|bool {
    array_push($permissions, 'access toolbar', 'access administration pages');
    return $this->drupalCreateUser($permissions);
  }

  /**
   * Refresh the toolbar domain menu links list.
   */
  protected function refreshPage(): void {
    $this->drupalGet(self::TEST_PATH);
    $this->toolbarDomainMenuLinks = $this->xpath($this->domainMenuLinksXpath);
  }

  /**
   * Test if domains and the menu links have the same weight.
   */
  protected function testMenuLinksSizeAndOrder(): void {
    $domains = $this->getDomainsSorted();
    $menu_links = $this->toolbarDomainMenuLinks;

    $this->assertCount(count($domains), $menu_links);

    foreach ($domains as $domain) {
      $this->assertEquals($domain->label(), current($menu_links)->getText(), 'Domain and menu link labels do not match.');
      next($menu_links);
    }
  }

}
