<?php

namespace Drupal\domain_path;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\domain\DomainNegotiatorInterface;
use Drupal\path_alias\AliasManagerInterface;

/**
 * Override the alias manager to use domain_path records.
 */
class DomainPathAliasManager implements AliasManagerInterface {

  /**
   * The domain_path.settings config object.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * The cache key to use when caching paths.
   *
   * @var string
   */
  protected $cacheKey;

  /**
   * Whether the cache needs to be written.
   *
   * @var bool
   */
  protected $cacheNeedsWriting = FALSE;

  /**
   * Holds the map of path lookups per language.
   *
   * @var array
   */
  protected $lookupMap = [];

  /**
   * Holds an array of aliases for which no path was found.
   *
   * @var array
   */
  protected $noPath = [];

  /**
   * Holds an array of paths that have no alias.
   *
   * @var array
   */
  protected $noAlias = [];

  /**
   * Whether preloaded path lookups has already been loaded.
   *
   * @var array
   */
  protected $langcodePreloaded = [];

  /**
   * Holds an array of previously looked up paths for the current request path.
   *
   * This will only get populated if a cache key has been set, which for example
   * happens if the alias manager is used in the context of a request.
   *
   * @var array
   */
  protected $preloadedPathLookups = FALSE;

  public function __construct(
    protected AliasManagerInterface $inner,
    protected LanguageManagerInterface $languageManager,
    protected CacheBackendInterface $cache,
    protected TimeInterface $time,
    protected DomainNegotiatorInterface $domainNegotiator,
    protected EntityTypeManagerInterface $entityTypeManager,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->config = $config_factory->get('domain_path.settings');
  }

  /**
   * Returns the domain path storage.
   *
   * @return \Drupal\Core\Entity\EntityStorageInterface
   *   The domain path storage.
   */
  protected function getDomainPathStorage() {
    return $this->entityTypeManager->getStorage('domain_path');
  }

  /**
   * {@inheritdoc}
   */
  public function getPathByAlias($alias, $langcode = NULL) {
    $active = $this->domainNegotiator->getActiveDomain();

    // If there is no active domain or if we already know that there are
    // no domain paths for this alias, simply return the default path.
    if (!$active || empty($alias) || !empty($this->noPath[$langcode][$alias])) {
      return $this->inner->getPathByAlias($alias, $langcode);
    }

    // Look for the domain alias within the cached map.
    if (isset($this->lookupMap[$langcode])
      && ($path = array_search($alias, $this->lookupMap[$langcode]))) {
      return $path;
    }

    $properties = [
      'alias' => $alias,
      'domain_id' => $active->id(),
    ];
    $domain_paths = $this->getDomainPathStorage()->loadByProperties($properties);

    // https://git.drupalcode.org/project/drupal/-/blob/9.2.x/core/modules/path_alias/src/PathProcessor/AliasPathProcessor.php#L36
    // didn't pass the $langcode.
    $method = $this->config->get('language_method') ?: LanguageInterface::TYPE_CONTENT;
    $langcode = $langcode ?: $this->languageManager->getCurrentLanguage($method)->getId();
    if ($langcode === NULL) {
      // @todo keep a "zxx -> Not applicable" in record for language negotiation failed?
      // LANGCODE_NOT_SPECIFIED = 'und'
      $langcode = LanguageInterface::LANGCODE_NOT_SPECIFIED;
      // Return the first record when language negotiation failed at this
      // moment.
      $domain_path = reset($domain_paths);
      if ($domain_path) {
        $path = $domain_path->getSource();
        $this->lookupMap[$langcode][$path] = $alias;
        return $path;
      }
    }
    else {
      foreach ($domain_paths as $domain_path) {
        if ($domain_path->getLanguageCode() === $langcode) {
          $path = $domain_path->getSource();
          $this->lookupMap[$langcode][$path] = $alias;
          return $path;
        }
      }
    }

    // We can't record anything into $this->lookupMap because we didn't find any
    // domain paths for this alias. Thus cache to $this->noPath.
    $this->noPath[$langcode][$alias] = TRUE;

    return $this->inner->getPathByAlias($alias, $langcode);
  }

  /**
   * {@inheritdoc}
   */
  public function getAliasByPath($path, $langcode = NULL) {
    if (!str_starts_with($path, '/')) {
      throw new \InvalidArgumentException(sprintf('Source path %s has to start with a slash.', $path));
    }

    // The root path is always aliased to itself.
    if ($path === '/') {
      return $path;
    }

    $active = $this->domainNegotiator->getActiveDomain();
    if (!$active) {
      return $this->inner->getAliasByPath($path, $langcode);
    }

    $method = $this->config->get('language_method') ?: LanguageInterface::TYPE_CONTENT;
    $langcode = $langcode ?: $this->languageManager->getCurrentLanguage($method)->getId();

    // During the first call to this method per language, load the expected
    // paths for the page from cache.
    if (empty($this->langcodePreloaded[$langcode])) {
      $this->langcodePreloaded[$langcode] = TRUE;
      $this->lookupMap[$langcode] = [];

      // Load the cached paths that should be used for preloading. This only
      // happens if a cache key has been set.
      if ($this->preloadedPathLookups === FALSE) {
        $this->preloadedPathLookups = [];
        if ($this->cacheKey) {
          if ($cached = $this->cache->get($this->cacheKey)) {
            $this->preloadedPathLookups = $cached->data;
          }
          else {
            $this->cacheNeedsWriting = TRUE;
          }
        }
      }

      // Load paths from cache.
      if (!empty($this->preloadedPathLookups[$langcode])) {
        $this->lookupMap[$langcode] =
          $this->preloadPathAlias($this->preloadedPathLookups[$langcode], $active, $langcode);
        // Keep a record of paths with no alias to avoid querying twice.
        $this->noAlias[$langcode] =
          array_flip(array_diff(
            $this->preloadedPathLookups[$langcode],
            array_keys($this->lookupMap[$langcode]),
          ));
      }
    }

    // If we already know that there are no aliases for this path simply return.
    if (!empty($this->noAlias[$langcode][$path])) {
      return $path;
    }

    // If the alias has already been loaded, return it from static cache.
    if (isset($this->lookupMap[$langcode][$path])) {
      return $this->lookupMap[$langcode][$path];
    }

    // Try to load alias from storage.
    if ($path_alias = $this->lookupBySystemPath($path, $active, $langcode)) {
      $this->lookupMap[$langcode][$path] = $path_alias->getAlias();
      return $path_alias->getAlias();
    }

    // We can't record anything into $this->lookupMap because we didn't find any
    // aliases for this path. Thus cache to $this->noAlias.
    $this->noAlias[$langcode][$path] = TRUE;

    return $path;
  }

  /**
   * Preload aliases for some system paths for a specific domain and language.
   *
   * Loads domain_path entities matching the provided paths, active domain and
   * language, and returns an associative array mapping source paths to aliases.
   *
   * @param array $paths
   *   An array of source paths to preload.
   * @param \Drupal\domain\DomainInterface $active
   *   The active domain object.
   * @param string $langcode
   *   The language code to filter domain path records.
   *
   * @return array
   *   Associative array where the keys are source paths and the values are
   *   aliases.
   */
  protected function preloadPathAlias(array $paths, $active, string $langcode): array {
    $properties = [
      'source' => $paths,
      'language' => $langcode,
      'domain_id' => $active->id(),
    ];
    $domain_path_aliases = $this->getDomainPathStorage()->loadByProperties($properties);
    $aliases = [];
    foreach ($domain_path_aliases as $result) {
      $aliases[$result->getSource()] = $result->getAlias();
    }
    return $aliases;
  }

  /**
   * Loads a domain_path entity for a given system path, domain and language.
   *
   * Searches the domain_path storage for an entity whose source path matches
   * the provided system path, belongs to the provided active domain, and has
   * the given language code. Returns the first matching entity or NULL if none
   * is found.
   *
   * @param string $path
   *   The system/source path (must start with a slash).
   * @param \Drupal\domain\DomainInterface $active
   *   The active domain object.
   * @param string $langcode
   *   The language code to filter domain path records.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The first matching domain_path entity, or NULL if no matching entity was
   *   found.
   */
  protected function lookupBySystemPath($path, $active, $langcode) {
    $properties = [
      'source' => $path,
      'language' => $langcode,
      'domain_id' => $active->id(),
    ];
    $path_aliases = $this->getDomainPathStorage()->loadByProperties($properties);
    return reset($path_aliases);
  }

  /**
   * {@inheritdoc}
   */
  public function cacheClear($source = NULL) {
    // Clear the cache of the inner alias manager.
    $this->inner->cacheClear($source);
    // Note this method does not flush the preloaded path lookup cache. This is
    // because if a path is missing from this cache, it still results in the
    // alias being loaded correctly, only less efficiently.
    if ($source) {
      foreach (array_keys($this->lookupMap) as $lang) {
        unset($this->lookupMap[$lang][$source]);
      }
    }
    else {
      $this->lookupMap = [];
    }
    $this->noPath = [];
    $this->noAlias = [];
    $this->langcodePreloaded = [];
    $this->preloadedPathLookups = [];
  }

  /**
   * This method is part of AliasManager, but not AliasManagerInterface.
   */
  public function setCacheKey($key) {
    // Set the cache key for the inner alias manager.
    if (method_exists($this->inner, 'setCacheKey')) {
      $this->inner->setCacheKey($key);
    }
    // Prefix the cache key to avoid clashes with other caches.
    $active = $this->domainNegotiator->getActiveDomain();
    if ($active) {
      $this->cacheKey = 'preload-domain-paths:' . $active->id() . ':' . $key;
    }
  }

  /**
   * This method is part of AliasManager, but not AliasManagerInterface.
   */
  public function writeCache() {
    // Write the cache for the inner alias manager.
    if (method_exists($this->inner, 'writeCache')) {
      $this->inner->writeCache();
    }
    // Check if the paths for this page were loaded from cache in this request
    // to avoid writing to cache on every request.
    if ($this->cacheNeedsWriting && !empty($this->cacheKey)) {
      // Start with the preloaded path lookups, so that cached entries for
      // other languages will not be lost.
      $path_lookups = $this->preloadedPathLookups ?: [];
      foreach ($this->lookupMap as $langcode => $lookups) {
        $path_lookups[$langcode] = array_keys($lookups);
        if (!empty($this->noAlias[$langcode])) {
          $path_lookups[$langcode] =
            array_merge($path_lookups[$langcode], array_keys($this->noAlias[$langcode]));
        }
      }

      $twenty_four_hours = 60 * 60 * 24;
      $this->cache->set($this->cacheKey, $path_lookups, $this->time->getRequestTime() + $twenty_four_hours);
    }
  }

}
