<?php

namespace Drupal\domain_path_pathauto;

use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\domain_path\DomainPathHelper;

/**
 * DomainPathauto helper service.
 */
class DomainPathautoHelper {

  use DependencySerializationTrait;
  use StringTranslationTrait;

  /**
   * The configuration.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $config;

  /**
   * DomainPathautoHelper constructor.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected DomainPathautoGenerator $domainPathautoGenerator,
    ConfigFactoryInterface $config_factory,
    protected DomainPathHelper $domainPathHelper,
  ) {
    $this->config = $config_factory->get('domain_path.settings');
  }

  /**
   * The domain path_auto form element for the entity form.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   Related entity.
   * @param array $keys
   *   The keys to the domain path form element.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function alterEntityForm(array &$form, FormStateInterface $form_state, ContentEntityInterface $entity, array $keys) {
    // We need to use it in validation callback.
    $form_state->addBuildInfo('element_keys', $keys);
    $domains = $this->entityTypeManager->getStorage('domain')
      ->loadMultipleSorted();
    // When setting states of checkboxes it's matter whether we have enabled or
    // disabled default pathauto UI. Depending on it, we have different paths.
    $input_selector_prefix = 'input[name="domain_path';
    if (!$this->config->get('hide_path_alias_ui')) {
      $input_selector_prefix = 'input[name="path[0][domain_path]';
    }
    $build_info = $form_state->getBuildInfo();
    foreach ($domains as $domain_id => $domain) {
      $pathautoEnabled = $this->domainPathautoGenerator->domainPathPathautoGenerationIsEnabled($entity, $domain->id());
      if ($entity->isNew() || $entity->isNewTranslation()) {
        $pathautoEnabled = $pathautoEnabled || $this->domainPathautoGenerator->getPatternByEntity($entity);
      }
      // See https://git.drupalcode.org/project/pathauto/-/blob/8.x-1.x/src/PathautoWidget.php#L42
      // Generate checkboxes per each domain.
      if ($build_info['pathauto_checkbox'] && NestedArray::getValue($form, $keys)) {
        NestedArray::setValue($form, array_merge($keys, [$domain_id, 'pathauto']), [
          '#type' => 'checkbox',
          '#title' => $this->t('Generate automatic URL alias for @domain', ['@domain' => Html::escape(rtrim($domain->getPath(), '/'))]),
          '#default_value' => $pathautoEnabled,
          '#weight' => -1,
        ]);
      }
      // Disable form element if the "delete-checkbox" is active or automatic
      // creation of alias is checked.
      NestedArray::setValue($form, array_merge($keys, [$domain_id, 'path', '#states']), [
        'disabled' => [
          [$input_selector_prefix . '[domain_path_delete]"]' => ['checked' => TRUE]],
          'OR',
          [$input_selector_prefix . '[' . $domain_id . '][pathauto]"]' => ['checked' => TRUE]],
        ],
      ]);
    }
    $form['#validate'][] = [$this, 'validateAlteredForm'];
    if (!empty($form['actions'])) {
      if (array_key_exists('submit', $form['actions'])) {
        $form['actions']['submit']['#submit'][] = [
          $this,
          'submitAlteredEntityForm',
        ];
      }
    }
    else {
      // If no actions we just tack it on to the form submit handlers.
      $form['#submit'][] = [$this, 'submitAlteredEntityForm'];
    }
  }

  /**
   * Validation handler.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function validateAlteredForm(array &$form, FormStateInterface $form_state) {
    // Not used anymore.  The code has been refactored into the
    // DomainPathHelper::validateEntityForm method.
  }

  /**
   * Submit handler.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException|\Drupal\Core\Entity\EntityStorageException
   */
  public function submitAlteredEntityForm(array $form, FormStateInterface $form_state) {
    $path_values = $form_state->getValue('path');
    $domain_path_values = $this->config->get('hide_path_alias_ui')
      ? $form_state->getValue('domain_path')
      : $path_values[0]['domain_path'];

    // If not set to delete, then save changes.
    if (!empty($domain_path_values['domain_path_delete'])) {
      return;
    }
    unset($domain_path_values['domain_path_delete']);

    $entity = $form_state->getFormObject()->getEntity();
    $entity_system_path = '/' . $entity->toUrl()->getInternalPath();
    $properties = [
      'source' => $entity_system_path,
      'language' => $entity->language()->getId(),
    ];

    // Check domain access settings if they are on the form.
    $domain_access = $this->domainPathHelper->getDomainAccess($form, $form_state);

    $domain_path_storage = $this->entityTypeManager->getStorage('domain_path');

    foreach ($domain_path_values as $domain_id => $domain_path_data) {

      $alias = rtrim(trim($domain_path_data['path']), " \\/");
      if (isset($domain_path_data['pathauto']) && $domain_path_data['pathauto']) {
        // Generate alias using pathauto.
        $alias = $this->domainPathautoGenerator->createEntityAlias($entity, 'return', $domain_id);
        // Remember pathauto default enabled setting.
        $this->domainPathautoGenerator->setDomainPathPathautoState($entity, $domain_id, TRUE);
      }
      else {
        // Delete pathauto default enabled setting.
        $this->domainPathautoGenerator->deleteDomainPathPathautoState($entity, $domain_id);
      }

      // Get the existing domain path for this domain if it exists.
      $properties['domain_id'] = $domain_id;
      $domain_paths = $domain_path_storage->loadByProperties($properties);
      $domain_path = $domain_paths ? reset($domain_paths) : NULL;

      // Check domain access.
      $domain_has_access = $domain_access === TRUE || isset($domain_access[$domain_id]);

      // We don't want to save the alias if the domain path field is empty
      // or if the domain doesn't have access to this entity.
      if (!$alias || !$domain_has_access) {
        // Delete the existing domain path.
        if ($domain_path) {
          $domain_path->delete();
        }
        continue;
      }

      // Create or update the domain path.
      $properties_map = [
        'alias' => $alias,
        'domain_id' => $domain_id,
      ] + $properties;
      if (!$domain_path) {
        $domain_path = $this->entityTypeManager->getStorage('domain_path')
          ->create(['type' => 'domain_path']);
        foreach ($properties_map as $field => $value) {
          $domain_path->set($field, $value);
        }
        $domain_path->save();
      }
      elseif ($domain_path->get('alias')->value !== $alias) {
        $domain_path->set('alias', $alias);
        $domain_path->save();
      }
    }
  }

}
