<?php

namespace Drupal\Tests\domain_path\Functional;

use Drupal\Core\Url;
use Drupal\node\Entity\Node;

/**
 * Verifies domain_path fields visibility.
 *
 * With domain_access enabled, while Domain Access fields are hidden from the
 * node form.
 *
 * Scenarios covered:
 * - Editor (single-domain access, no "publish to any domain") sees only the
 *   path field(s) for assigned domain(s) they can publish to and can save an
 *   alias only for those.
 * - Admin (has "publish to any domain") sees path fields for all assigned
 *   domains and can save aliases for each assigned domain.
 *
 * @see https://www.drupal.org/i/3375368
 *
 * @group domain_path
 */
class DomainPathVisibilityWithDomainAccessTest extends DomainPathTestBase {

  /**
   * Ensure Domain Access fields are hidden from the form display.
   */
  protected function hideDomainAccessFieldsOnPageBundle(): void {
    // Remove Domain Access fields from the form display of the 'page' bundle.
    // We remove both single and all affiliates field if present.
    $form_display = \Drupal::service('entity_display.repository')
      ->getFormDisplay('node', 'page');
    foreach (['field_domain_access', 'field_domain_all_affiliates'] as $field) {
      if ($form_display->getComponent($field)) {
        $form_display->removeComponent($field)->save();
      }
    }
  }

  /**
   * Assign a node to specific domains using Domain Access fields.
   */
  protected function assignNodeToDomains(Node $node, array $domain_ids, bool $all_affiliates = FALSE): void {
    // Ensure fields exist; Domain Access module provides them on standard
    // profile.
    if ($all_affiliates) {
      $node->set('field_domain_all_affiliates', 1);
    }
    if ($node->hasField('field_domain_access')) {
      $node->set('field_domain_access', $domain_ids);
    }
    $node->save();
  }

  /**
   * Build username based on domain id for clarity.
   */
  protected function usernameForDomain(string $domain_id): string {
    return 'editor_' . preg_replace('/[^a-z0-9_]+/i', '_', $domain_id);
  }

  /**
   * Creates an editor limited to a single domain.
   */
  protected function createSingleDomainEditor(string $domain_id) {
    // Create a user with publish permission only for one domain; omit
    // 'publish to any domain'. In Domain Access, per-domain permissions are
    // stored as user fields (field_domain_access) and evaluated by the
    // DomainAccessManager. We assign the account to the domain and grant
    // standard content perms.
    $perms = [
      'access content',
      'create page content',
      'edit own page content',
      'edit domain path entity',
      'add domain paths',
      'edit domain paths',
    ];
    $user = $this->drupalCreateUser($perms, $this->usernameForDomain($domain_id));
    // Assign user to the specific domain (no global publish permission).
    if ($user->hasField('field_domain_access')) {
      $user->set('field_domain_access', [$domain_id]);
      $user->save();
    }
    return $user;
  }

  /**
   * Creates an admin with publish to any domain.
   */
  protected function createAdminAllDomains() {
    return $this->drupalCreateUser([
      'bypass node access',
      'administer content types',
      'administer users',
      'administer node fields',
      'administer node display',
      'administer domains',
      'administer url aliases',
      'administer domain paths',
      'edit domain path entity',
      'add domain paths',
      'edit domain paths',
      'delete domain paths',
      'publish to any domain',
    ]);
  }

  /**
   * Tests visibility and saving behavior for editor and admin.
   */
  public function testVisibilityAndSaving(): void {
    // Basic module setup from base class already enabled
    // domain_path + domain_access.
    // Ensure settings allow nodes.
    $this->config('domain_path.settings')->set('entity_types', ['node' => TRUE])->save();

    // Hide domain access fields from the form.
    $this->hideDomainAccessFieldsOnPageBundle();

    // Prepare two domains; base created in setUp of parent. Use first two.
    $domain_ids = array_keys($this->domains);
    $domain_a = $domain_ids[0];
    $domain_b = $domain_ids[1] ?? $domain_ids[0];

    // Create users first so we can author the node by the editor, ensuring
    // edit access for the editor (has only "edit own page content").
    $editor = $this->createSingleDomainEditor($domain_a);
    $admin = $this->createAdminAllDomains();

    // Create a node authored by the editor and assign only to domain A.
    $node = Node::create([
      'type' => 'page',
      'title' => $this->randomMachineName(),
      'body' => [
        'value' => $this->randomMachineName(20),
      ],
      'status' => 1,
      'uid' => $editor->id(),
    ]);
    $node->save();
    $this->assignNodeToDomains($node, [$domain_a]);

    // 1) Editor: sees only domain A path field; not domain B.
    $this->drupalLogin($editor);
    $this->drupalGet(Url::fromRoute('entity.node.edit_form', ['node' => $node->id()]));
    $session = $this->assertSession();

    // Domain Access fields must be absent.
    $this->findNoField('field_domain_access[' . $domain_a . ']');
    $this->findNoField('field_domain_access[' . $domain_b . ']');

    // domain_path field for assigned A exists, for B does not.
    $session->fieldExists('domain_path[' . $domain_a . '][path]');
    $session->fieldNotExists('domain_path[' . $domain_b . '][path]');

    // Enter alias for A and try for B (should not be present).
    $alias_a = '/' . $this->randomMachineName(8);
    $this->fillField('domain_path[' . $domain_a . '][path]', $alias_a);
    $this->pressButton('Save');

    // Verify domain_path created only for A.
    $storage = \Drupal::entityTypeManager()->getStorage('domain_path');
    $this->assertCount(1, $storage->loadByProperties([
      'domain_id' => $domain_a,
      'alias' => $alias_a,
    ]));
    $this->assertCount(0, $storage->loadByProperties([
      'domain_id' => $domain_b,
    ]));

    // 2) Admin: sees path fields for all assigned domains. The node is only
    // assigned to A, so admin should see only A (since domain_path fields
    // should display for assigned domains). Now assign B as well and verify.
    $this->drupalLogin($admin);

    // Initially (assigned only A): B should not appear.
    $this->drupalGet('node/' . $node->id() . '/edit');
    $session->fieldExists('domain_path[' . $domain_a . '][path]');
    $session->fieldNotExists('domain_path[' . $domain_b . '][path]');

    // Assign B too and verify both appear for admin.
    $this->assignNodeToDomains($node, [$domain_a, $domain_b]);

    $this->drupalGet('node/' . $node->id() . '/edit');
    $session->fieldExists('domain_path[' . $domain_a . '][path]');
    $session->fieldExists('domain_path[' . $domain_b . '][path]');

    $alias_b = '/' . $this->randomMachineName(8);
    $this->fillField('domain_path[' . $domain_b . '][path]', $alias_b);
    $this->pressButton('Save');

    // Verify domain_path created for B as well.
    $this->assertCount(1, $storage->loadByProperties([
      'domain_id' => $domain_b,
      'alias' => $alias_b,
    ]));
  }

}
