<?php

namespace Drupal\Tests\domain_path_pathauto\Functional;

use Drupal\domain_path_pathauto\DomainPathautoState;
use Drupal\Tests\domain_path\Functional\DomainPathTestBase;
use Drupal\Tests\pathauto\Functional\PathautoTestHelperTrait;

/**
 * Tests the deletion of Domain Path Pathauto data.
 *
 * @group domain_path_pathauto
 */
class PathautoNodeDeleteTest extends DomainPathTestBase {

  use PathautoTestHelperTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'domain_path_pathauto',
    'pathauto',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $pattern = $this->createPattern('node', '/content/[node:title]');
    $this->addBundleCondition($pattern, 'node', 'page');
    $pattern->save();
  }

  /**
   * Tests node deletion and its effect on domain path and pathauto state.
   */
  public function testNodeDelete(): void {
    $title = $this->randomMachineName();
    $this->drupalGet('node/add/page');
    $this->assertSession()->statusCodeEquals(200);
    $this->fillField('title[0][value]', $title);
    $this->pressButton('Save');
    $this->assertSession()->pageTextContains($title);

    $node = $this->drupalGetNodeByTitle($title);
    $source = '/node/' . $node->id();

    // Verify domain_path entity exists for each domain.
    foreach ($this->domains as $domain) {
      $domain_id = $domain->id();
      $domain_paths = \Drupal::entityTypeManager()
        ->getStorage('domain_path')
        ->loadByProperties([
          'domain_id' => $domain_id,
          'source' => $source,
        ]);
      $this->assertCount(1, $domain_paths, "Domain path entity exists for domain $domain_id before deletion.");

      // Verify pathauto state exists in key_value store.
      $collection = 'domain_path_pathauto_state.' . $domain_id . '.node';
      $key = DomainPathautoState::getPathautoStateKey($node->id());
      $state = \Drupal::keyValue($collection)->get($key);
      $this->assertEquals(DomainPathautoState::CREATE, $state, "Pathauto state exists in key_value store for domain $domain_id before deletion.");
    }

    // Delete the node.
    $node->delete();

    // Verify domain_path entity and state are deleted for each domain.
    foreach ($this->domains as $domain) {
      $domain_id = $domain->id();
      $domain_paths = \Drupal::entityTypeManager()
        ->getStorage('domain_path')
        ->loadByProperties([
          'domain_id' => $domain_id,
          'source' => $source,
        ]);
      $this->assertCount(0, $domain_paths, "Domain path entity is deleted for domain $domain_id after node deletion.");

      $collection = 'domain_path_pathauto_state.' . $domain_id . '.node';
      $key = DomainPathautoState::getPathautoStateKey($node->id());
      $state = \Drupal::keyValue($collection)->get($key);
      $this->assertNull($state, "Pathauto state is deleted from key_value store for domain $domain_id after node deletion.");
    }
  }

}
