<?php

namespace Drupal\domain_path\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\ConfigTarget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\RedundantEditableConfigNamesTrait;
use Drupal\Core\Language\LanguageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure domain_path settings for this site.
 *
 * @package Drupal\domain_path\Form
 * @ingroup domain_path
 */
class DomainPathSettingsForm extends ConfigFormBase {

  use RedundantEditableConfigNamesTrait;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($config_factory, $typed_config_manager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'domain_path_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $entity_types = [];

    // Get all applicable entity types.
    foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $entity_type) {
      if (is_subclass_of($entity_type->getClass(), FieldableEntityInterface::class) && $entity_type->hasLinkTemplate('canonical')) {
        if ($entity_type_id === 'domain_path' || $entity_type_id === 'domain_path_redirect') {
          continue;
        }
        $entity_types[$entity_type_id] = $entity_type->getLabel();
      }
    }
    $form['entity_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Enabled entity types for domain paths'),
      '#config_target' => new ConfigTarget('domain_path.settings', 'entity_types', toConfig: fn ($value) => array_values(array_filter($value))),
      '#options' => $entity_types,
    ];

    $form['language_method'] = [
      '#type' => 'radios',
      '#title' => $this->t('The method of language detection'),
      '#options' => [
        LanguageInterface::TYPE_CONTENT => $this->t('Content language'),
        LanguageInterface::TYPE_INTERFACE => $this->t('Interface text language'),
        LanguageInterface::TYPE_URL => $this->t('Language from URLs'),
      ],
      '#config_target' => 'domain_path.settings:language_method',
      '#description' => $this->t('If you enabled multilingual content for certain domains, you need to set it according to your language settings.'),
    ];
    $options = [
      'name' => $this->t('The domain display name'),
      'hostname' => $this->t('The raw hostname'),
      'url' => $this->t('The domain base URL'),
    ];

    $form['alias_title'] = [
      '#type' => 'radios',
      '#title' => $this->t('Domain path alias title'),
      '#config_target' => 'domain_path.settings:alias_title',
      '#options' => $options,
      '#description' => $this->t('Select the text to display for each field in entity edition.'),
    ];

    $form['hide_path_alias_ui'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Hide the default URL alias UI'),
      '#config_target' => 'domain_path.settings:hide_path_alias_ui',
      '#description' => $this->t('Hide the default URL alias options from the UI to avoid the confusion. Domain path will replace the default URL alias with each individual domains alias'),
    ];

    return parent::buildForm($form, $form_state);
  }

}
