<?php

namespace Drupal\Tests\domain_path\Functional;

/**
 * Tests the domain path node aliases saving from edit form with autocomplete.
 *
 * Duplicates DomainPathNodeAliasTest but checks the Domain Settings tab uses
 * an autocomplete widget instead of checkboxes for field_domain_access.
 *
 * @group domain_path
 */
class DomainPathNodeAliasAutocompleteTest extends DomainPathTestBase {

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Switch the Domain Access field widget to standard entity reference
    // autocomplete (non-tags). With unlimited cardinality this renders an
    // indexed input [0][target_id] and an "Add another item" control.
    /** @var \Drupal\Core\Entity\Display\EntityDisplayInterface $display */
    $display = \Drupal::entityTypeManager()
      ->getStorage('entity_form_display')
      ->load('node.page.default');
    if ($display) {
      $display->setComponent('field_domain_access', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 40,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => 60,
          'placeholder' => '',
        ],
      ])->save();
    }

    // Refresh field definitions after the configuration update.
    \Drupal::service('entity_field.manager')->clearCachedFieldDefinitions();
  }

  /**
   * Test domain path node aliases with domain access as autocomplete.
   */
  public function testDomainPathNodeAliasesFillWithAutocomplete() {
    // With autocomplete not filled, only the active/current domain is assigned
    // by default, so only that alias should be persisted.
    $active_domain_id = \Drupal::service('domain.negotiator')->getActiveDomain()->id();

    // Set content title and body from Node form.
    $edit = [];
    $edit['title[0][value]'] = $this->randomMachineName(8);
    $edit['body[0][value]'] = $this->randomMachineName(16);

    // Open the node add form and ensure the Domain Settings field exists as
    // an autocomplete input instead of individual checkboxes.
    $this->drupalGet('node/add/page');
    // For standard entity reference autocomplete (non-tags) with unlimited
    // cardinality, the input name is field_domain_access[0][target_id].
    $this->assertSession()->fieldExists('field_domain_access[0][target_id]');

    // When using the autocomplete widget, no domain path fields should be
    // visible until the content has been created.
    foreach ($this->domains as $domain) {
      $this->assertSession()->fieldNotExists('domain_path[' . $domain->id() . '][alias]');
    }

    // Submit the form with domain-specific aliases.
    $this->submitForm($edit, t('Save'));

    // Open the node edit form.
    $this->drupalGet('node/1/edit');

    // When using the autocomplete widget, no domain path fields should be
    // visible until the content has been created.
    foreach ($this->domains as $domain) {
      if ($domain->id() === $active_domain_id) {
        $this->assertSession()->fieldExists('domain_path[' . $domain->id() . '][alias]');
      }
      else {
        $this->assertSession()->fieldNotExists('domain_path[' . $domain->id() . '][alias]');
      }
    }

    // Set path from Node form.
    $domain_paths_check = [];
    foreach ($this->domains as $domain) {
      $domain_alias_value = '/' . $this->randomMachineName(8);
      $domain_paths_check[$domain->id()] = $domain_alias_value;
      // Only the active domain has been added to the domain access field.
      // So only the corresponding domain path can be created.
      if ($domain->id() === $active_domain_id) {
        $edit['domain_path[' . $domain->id() . '][alias]'] = $domain_alias_value;
      }
    }

    // Submit the form with domain-specific aliases.
    $this->submitForm($edit, t('Save'));

    $storage = \Drupal::service('entity_type.manager')->getStorage('domain_path');
    foreach ($domain_paths_check as $domain_id => $domain_alias_value) {
      $expected_count = ($domain_id === $active_domain_id) ? 1 : 0;
      $this->assertCount($expected_count, $storage->loadByProperties([
        'domain_id' => $domain_id,
        'alias' => $domain_alias_value,
      ]));
    }

    // Check values on node form.
    $this->drupalGet('node/1/edit');
    foreach ($domain_paths_check as $domain_id => $domain_alias_value) {
      if ($domain_id === $active_domain_id) {
        $this->assertSession()->fieldValueEquals(
          'domain_path[' . $domain_id . '][alias]', $domain_alias_value);
      }
    }
  }

}
