<?php

namespace Drupal\domain_path;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\domain\DomainNegotiatorInterface;
use Drupal\path_alias\AliasManagerInterface;

/**
 * Override the alias manager to use domain_path records.
 */
class DomainPathAliasManager implements AliasManagerInterface, DomainAliasManagerInterface {

  /**
   * The domain_path.settings config object.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * The cache key to use when caching paths.
   *
   * @var string
   */
  protected $cacheKey;

  /**
   * Whether the cache needs to be written.
   *
   * @var bool
   */
  protected $cacheNeedsWriting = FALSE;

  /**
   * Holds the map of path lookups per language.
   *
   * @var array
   */
  protected $lookupMap = [];

  /**
   * Holds an array of aliases for which no path was found.
   *
   * @var array
   */
  protected $noPath = [];

  /**
   * Holds an array of paths that have no alias.
   *
   * @var array
   */
  protected $noAlias = [];

  /**
   * Whether preloaded path lookups has already been loaded.
   *
   * @var array
   */
  protected $langcodePreloaded = [];

  /**
   * Holds an array of previously looked up paths for the current request path.
   *
   * This will only get populated if a cache key has been set, which for example
   * happens if the alias manager is used in the context of a request.
   *
   * @var array
   */
  protected $preloadedPathLookups = FALSE;

  public function __construct(
    protected AliasManagerInterface $inner,
    protected DomainAliasRepository $domainAliasRepository,
    protected LanguageManagerInterface $languageManager,
    protected CacheBackendInterface $cache,
    protected TimeInterface $time,
    protected DomainNegotiatorInterface $domainNegotiator,
    protected EntityTypeManagerInterface $entityTypeManager,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->config = $config_factory->get('domain_path.settings');
  }

  /**
   * Returns the domain path storage.
   *
   * @return \Drupal\Core\Entity\EntityStorageInterface
   *   The domain path storage.
   */
  protected function getDomainPathStorage() {
    return $this->entityTypeManager->getStorage('domain_path');
  }

  /**
   * {@inheritdoc}
   */
  public function getPathByAlias($alias, $langcode = NULL) {
    $active = $this->domainNegotiator->getActiveDomain();
    if ($active) {
      return $this->getDomainPathByAlias($alias, $active->id(), $langcode);
    }
    else {
      return $this->inner->getPathByAlias($alias, $langcode);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getDomainPathByAlias($alias, $domain_id, $langcode = NULL) {
    // If no language is explicitly specified we default to the language
    // corresponding to the method defined in the configuration.
    $method = $this->config->get('language_method') ?: LanguageInterface::TYPE_CONTENT;
    $langcode = $langcode ?: $this->languageManager->getCurrentLanguage($method)->getId();

    // If we already know that there are no domain paths for this alias,
    // simply return the default path.
    if (empty($alias) || !empty($this->noPath[$domain_id][$langcode][$alias])) {
      return $alias;
    }

    // Look for the alias within the cached map.
    if (isset($this->lookupMap[$domain_id][$langcode])
      && ($path = array_search($alias, $this->lookupMap[$domain_id][$langcode]))) {
      return $path;
    }

    // Look for path in storage.
    if ($path_alias = $this->domainAliasRepository->lookupByAlias($alias, $domain_id, $langcode)) {
      $this->lookupMap[$domain_id][$langcode][$path_alias['source']] = $alias;
      return $path_alias['source'];
    }

    // Fall back to core aliasing.
    $path = $this->inner->getPathByAlias($alias, $langcode);

    if ($path === $alias) {
      // If no valid alias was found, cache it to avoid querying it again.
      $this->noPath[$domain_id][$langcode][$alias] = TRUE;
    }
    else {
      $this->lookupMap[$domain_id][$langcode][$path] = $alias;
      return $path;
    }

    return $alias;
  }

  /**
   * {@inheritdoc}
   */
  public function getAliasByPath($path, $langcode = NULL) {
    $active = $this->domainNegotiator->getActiveDomain();
    if ($active) {
      return $this->getDomainAliasByPath($path, $active->id(), $langcode);
    }
    else {
      return $this->inner->getAliasByPath($path, $langcode);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getDomainAliasByPath($path, $domain_id, $langcode = NULL) {
    if (!str_starts_with($path, '/')) {
      throw new \InvalidArgumentException(sprintf('Source path %s has to start with a slash.', $path));
    }

    // The root path is always aliased to itself.
    if ($path === '/') {
      return $path;
    }

    $method = $this->config->get('language_method') ?: LanguageInterface::TYPE_CONTENT;
    $langcode = $langcode ?: $this->languageManager->getCurrentLanguage($method)->getId();

    // During the first call to this method per language, load the expected
    // paths for the page from cache.
    if (empty($this->langcodePreloaded[$domain_id][$langcode])) {
      $this->langcodePreloaded[$domain_id][$langcode] = TRUE;
      $this->lookupMap[$domain_id][$langcode] = [];

      // Load the cached paths that should be used for preloading. This only
      // happens if a cache key has been set.
      if ($this->preloadedPathLookups === FALSE) {
        $this->preloadedPathLookups = [];
        if ($this->cacheKey) {
          if ($cached = $this->cache->get($this->cacheKey . ':' . $domain_id)) {
            $this->preloadedPathLookups[$domain_id] = $cached->data;
          }
          else {
            $this->cacheNeedsWriting = TRUE;
          }
        }
      }

      // Load paths from cache.
      if (!empty($this->preloadedPathLookups[$domain_id][$langcode])) {
        $this->lookupMap[$domain_id][$langcode] =
          $this->domainAliasRepository->preloadPathAlias(
            $this->preloadedPathLookups[$domain_id][$langcode], $domain_id, $langcode);
        // Keep a record of paths with no alias to avoid querying twice.
        $this->noAlias[$domain_id][$langcode] =
          array_flip(array_diff(
            $this->preloadedPathLookups[$domain_id][$langcode],
            array_keys($this->lookupMap[$domain_id][$langcode]),
          ));
      }
    }

    // If we already know that there are no aliases for this path simply return.
    if (!empty($this->noAlias[$domain_id][$langcode][$path])) {
      return $path;
    }

    // If the alias has already been loaded, return it from static cache.
    if (isset($this->lookupMap[$domain_id][$langcode][$path])) {
      return $this->lookupMap[$domain_id][$langcode][$path];
    }

    // Try to load alias from storage.
    if ($path_alias = $this->domainAliasRepository->lookupBySystemPath($path, $domain_id, $langcode)) {
      $this->lookupMap[$domain_id][$langcode][$path] = $path_alias['alias'];
      return $path_alias['alias'];
    }

    // Fallback to core aliasing.
    $alias = $this->inner->getAliasByPath($path, $langcode);

    if ($alias === $path) {
      // We can't record anything into $this->lookupMap because we didn't find
      // any aliases for this path. Thus cache to $this->noAlias.
      $this->noAlias[$domain_id][$langcode][$path] = TRUE;
    }
    else {
      $this->lookupMap[$domain_id][$langcode][$path] = $alias;
      return $alias;
    }

    return $path;
  }

  /**
   * {@inheritdoc}
   */
  public function cacheClear($source = NULL) {
    // Clear the cache of the inner alias manager.
    $this->inner->cacheClear($source);
    // Note this method does not flush the preloaded path lookup cache. This is
    // because if a path is missing from this cache, it still results in the
    // alias being loaded correctly, only less efficiently.
    if ($source) {
      foreach ($this->lookupMap as $domain_id => $domain_lookups) {
        foreach (array_keys($domain_lookups) as $lang) {
          unset($this->lookupMap[$domain_id][$lang][$source]);
        }
      }
    }
    else {
      $this->lookupMap = [];
    }
    $this->noPath = [];
    $this->noAlias = [];
    $this->langcodePreloaded = [];
    $this->preloadedPathLookups = [];
  }

  /**
   * This method is part of AliasManager, but not AliasManagerInterface.
   */
  public function setCacheKey($key) {
    // Set the cache key for the inner alias manager.
    if (method_exists($this->inner, 'setCacheKey')) {
      $this->inner->setCacheKey($key);
    }
    // Prefix the cache key to avoid clashes with other caches.
    $this->cacheKey = 'preload-domain-paths:' . $key;
  }

  /**
   * This method is part of AliasManager, but not AliasManagerInterface.
   */
  public function writeCache() {
    // Write the cache for the inner alias manager.
    if (method_exists($this->inner, 'writeCache')) {
      $this->inner->writeCache();
    }
    // Check if the paths for this page were loaded from cache in this request
    // to avoid writing to cache on every request.
    if ($this->cacheNeedsWriting && !empty($this->cacheKey)) {
      // Start with the preloaded path lookups, so that cached entries for
      // other languages will not be lost.
      $path_lookups = $this->preloadedPathLookups ?: [];
      foreach ($this->lookupMap as $domain_id => $domain_lookups) {
        foreach ($domain_lookups as $langcode => $lookups) {
          $path_lookups[$domain_id][$langcode] = array_keys($lookups);
          if (!empty($this->noAlias[$domain_id][$langcode])) {
            $path_lookups[$domain_id][$langcode] =
              array_merge($path_lookups[$domain_id][$langcode], array_keys($this->noAlias[$domain_id][$langcode]));
          }
        }
      }

      $twenty_four_hours = 60 * 60 * 24;
      foreach ($path_lookups as $domain_id => $domain_lookups) {
        $this->cache->set(
          $this->cacheKey . ':' . $domain_id,
          $domain_lookups,
          $this->time->getRequestTime() + $twenty_four_hours
        );
      }
    }
  }

}
