<?php

namespace Drupal\domain_path;

use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\domain_access\DomainAccessManager;
use Drupal\path_alias\AliasManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\domain\DomainInterface;
use Drupal\user\UserInterface;

/**
 * Helper class for domain path operations.
 */
class DomainPathHelper {

  use StringTranslationTrait;
  use DependencySerializationTrait;

  public const PATH_FIELD = 'domain_path';
  protected const DOMAIN_ACCESS = 'field_domain_access';
  protected const DOMAIN_ACCESS_ALL = 'field_domain_all_affiliates';

  /**
   * The configuration.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $config;

  public function __construct(
    protected AccountInterface $accountManager,
    protected DomainAliasManagerInterface $domainAliasManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected AliasManagerInterface $aliasManager,
    protected ModuleHandlerInterface $moduleHandler,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->config = $config_factory->get('domain_path.settings');
  }

  /**
   * The domain paths form element for the entity form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   Referenced entity.
   *
   * @return array
   *   Return the modified form array.
   */
  public function alterEntityForm(&$form, FormStateInterface $form_state, $entity) {
    $domain_access = self::getAlterFormDomainAccess($form, $form_state);

    $visible_paths = 0;
    $path_keys = Element::children($form[DomainPathHelper::PATH_FIELD]['widget']);
    foreach ($path_keys as $key) {
      $path_element =& $form[DomainPathHelper::PATH_FIELD]['widget'][$key];
      $alias_element =& $path_element['alias'];

      $domain_id = $path_element['domain_id']['#value'];

      if (isset($alias_element['#states']['disabled'])) {
        $alias_element['#states']['disabled'] = [
          $alias_element['#states']['disabled'],
          'or',
          ['input[name="domain_path_wrapper[domain_path_delete]"]' => ['checked' => TRUE]],
        ];
      }
      else {
        $alias_element['#states']['disabled'] =
          ['input[name="domain_path_wrapper[domain_path_delete]"]' => ['checked' => TRUE]];
      }

      $path = $alias_element['#default_value'] ?? '';
      if ($path > '') {
        $visible_paths += 1;
      }

      // If domain settings are on the page for this domain we only show if
      // it's checked. e.g. on the node form, we only show the domain path
      // field for domains we're publishing to.
      if (isset($form[self::DOMAIN_ACCESS]['widget']['#options'][$domain_id])) {
        $conditions = [];
        if ($form[self::DOMAIN_ACCESS]['widget']['#type'] === 'checkboxes') {
          $conditions['input[name="field_domain_access[' . $domain_id . ']"]'] = ['checked' => TRUE];
        }
        elseif ($form[self::DOMAIN_ACCESS]['widget']['#type'] === 'radios') {
          $conditions['input[name="field_domain_access"]'] = ['value' => $domain_id];
        }
        // #states conditions do not work correctly with multi-select widgets.
        elseif ($form[self::DOMAIN_ACCESS]['widget']['#type'] === 'select'
          && !($form[self::DOMAIN_ACCESS]['widget']['#multiple'] ?? FALSE)) {
          $conditions['select[name="field_domain_access"]'] = ['value' => $domain_id];
        }
        if (!empty($conditions)) {
          // Hide initially to prevent pre-#states flash.
          $path_element['#attributes']['style'] = 'display:none';
          $path_element['#states']['visible'] = [
            [
              $conditions,
              'or',
              [
                'input[name="' . self::DOMAIN_ACCESS_ALL . '[value]"]' => ['checked' => TRUE],
              ],
            ],
          ];
        }
      }
      // #options are available but do not contain the domain being processed,
      // so we hide the corresponding path field.
      elseif (isset($form[self::DOMAIN_ACCESS]['widget']['#options'])) {
        $path_element['#access'] = FALSE;
      }
      // Hide the path field when the user doesn't have access to this domain.
      elseif ($domain_access !== TRUE && !isset($domain_access[$domain_id])) {
        $path_element['#access'] = FALSE;
      }

    }

    // Container for domain path fields.
    $form[DomainPathHelper::PATH_FIELD] = [
      '#tree' => TRUE,
      '#type' => 'details',
      '#title' => $this->t('Domain-specific paths'),
      '#description' => $this->t('Override the default URL alias for individual domains.
      Alias must start with a slash.
      For example, type "/about" when writing an about page.'),
      // '#group' => 'path_settings',
      '#weight' => 110,
      '#open' => TRUE,
    ] + $form[DomainPathHelper::PATH_FIELD];

    // Add an option to delete all domain paths. This is just for convenience
    // so the user doesn't have to manually remove the paths from each domain.
    $form[DomainPathHelper::PATH_FIELD]['domain_path_delete'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Delete all aliases for this content'),
      '#description' => $this->t('Checking this box will delete all aliases for this content, avoiding the need to remove them individually.'),
      '#default_value' => FALSE,
      '#weight' => 1000,
    ];

    // We only need to enable the delete checkbox if we have at least two
    // visible defined domain paths. See https://www.drupal.org/i/3461258
    $show_delete = $visible_paths > 1;

    $form[DomainPathHelper::PATH_FIELD]['domain_path_delete']['#access'] = $show_delete;

    // Hide the default URL alias for better UI.
    if ($this->config->get('hide_path_alias_ui')) {
      unset($form['path']);
    }

    // Move the domain path field to the advanced group.
    if (
      isset($form['advanced'])
      && !isset($form[DomainPathHelper::PATH_FIELD]['#group'])
      && $this->config->get('use_advanced_group')
    ) {
      $form[DomainPathHelper::PATH_FIELD]['#group'] = 'advanced';
      // Move the domain path field below the domain group if present.
      $form['#after_build'][] = [$this, 'afterEntityFormBuild'];
    }

    // Add our validation and submit handlers.
    $form['#validate'][] = [$this, 'validateEntityForm'];
    if (!empty($form['actions']) && array_key_exists('submit', $form['actions'])) {
      $form['actions']['submit']['#submit'][] = [$this, 'submitEntityForm'];
    }
    else {
      // If no actions we just tack it on to the form submit handlers.
      $form['#submit'][] = [$this, 'submitEntityForm'];
    }

    return $form;
  }

  /**
   * Form #after_build callback.
   */
  public function afterEntityFormBuild(array $form, FormStateInterface $form_state): array {
    if (
      isset($form['domain']) &&
      isset($form[DomainPathHelper::PATH_FIELD])
    ) {
      $domain_group_weight = $form['domain']['#weight'];
      // Move the domain path field below the domain group.
      $form[DomainPathHelper::PATH_FIELD]['#weight'] = $domain_group_weight + 1;
    }
    return $form;
  }

  /**
   * Validation handler the domain paths element on the entity form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public function validateEntityForm(array &$form, FormStateInterface $form_state) {
    // No special validation needed at this time.
  }

  /**
   * Submit handler for the domain paths element on the entity form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public function submitEntityForm($form, FormStateInterface $form_state) {
    // Setup Variables.
    $path_wrapper_values = $form_state->getValue('path_wrapper');
    if (!empty($path_wrapper_values['domain_path_delete'])) {
      // Delete all domain path aliases.
      $entity = $form_state->getFormObject()->getEntity();
      $entity_system_path = '/' . $entity->toUrl()->getInternalPath();
      $properties = [
        'source' => $entity_system_path,
        'language' => $entity->language()->getId(),
      ];
      $domain_path_storage = $this->entityTypeManager->getStorage('domain_path');
      $domain_paths = $domain_path_storage->loadByProperties($properties);
      foreach ($domain_paths as $domain_path) {
        $domain_path->delete();
      }
    }
  }

  /**
   * Helper function for deleting domain paths from an entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity object.
   * @param bool $delete_all_translations
   *   Whether to delete domain paths for all translations (optional).
   *   Defaults to FALSE.
   */
  public function deleteEntityDomainPaths(EntityInterface $entity, $delete_all_translations = FALSE) {
    if ($this->domainPathsIsEnabled($entity)) {
      $properties_map = [
        'source' => '/' . $entity->toUrl()->getInternalPath(),
      ];
      if (!$delete_all_translations) {
        $properties_map['language'] = $entity->language()->getId();
      }
      $domain_paths = $this->entityTypeManager
        ->getStorage('domain_path')
        ->loadByProperties($properties_map);
      if ($domain_paths) {
        foreach ($domain_paths as $domain_path) {
          $domain_path->delete();
        }
      }
    }

    // Delete domain paths on domain delete.
    if ($entity instanceof DomainInterface) {
      $domain_paths = $this->entityTypeManager
        ->getStorage('domain_path')
        ->loadByProperties(['domain_id' => $entity->id()]);
      if ($domain_paths) {
        foreach ($domain_paths as $domain_path) {
          $domain_path->delete();
        }
      }
    }
  }

  /**
   * Helper function for retrieving configured entity types.
   *
   * @return array
   *   Returns array of configured entity types.
   */
  public function getConfiguredEntityTypes(): array {
    return $this->config->get('entity_types') ?? [];
  }

  /**
   * Check if domain paths is enabled for a given entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity object.
   *
   * @return bool
   *   Return TRUE or FALSE.
   */
  public function domainPathsIsEnabled(EntityInterface $entity) {
    return in_array($entity->getEntityTypeId(), $this->getConfiguredEntityTypes());
  }

  /**
   * Returns an array of processed domain access field values.
   */
  public function processDomainAccessField($field_values): array {
    $domain_access = [];
    if (is_array($field_values)) {
      // Handle case of autocomplete tags style.
      if (isset($field_values['target_id'])) {
        $field_values = $field_values['target_id'];
      }
      foreach ($field_values as $field_value) {
        if (!is_array($field_value)) {
          continue;
        }
        $domain_access[$field_value['target_id']] = $field_value['target_id'];
      }
    }

    return $domain_access;
  }

  /**
   * Determines the domain access settings based on the provided form and state.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return mixed
   *   Returns TRUE if the domain access module is not installed if domain
   *   access is enabled for all domains. Returns an array of target IDs if
   *   specific domain access is configured.
   */
  public function getDomainAccess(array $form, FormStateInterface $form_state) {
    if ($this->moduleHandler->moduleExists('domain_access')) {
      // If domain access is on for this form, we check the "all affiliates"
      // checkbox, otherwise we just assume it's not available on all domains.
      if (!empty($form[self::DOMAIN_ACCESS_ALL])
        && ($form_state->getValue(self::DOMAIN_ACCESS_ALL)['value'] ?? FALSE)) {
        $domain_access = TRUE;
      }
      // Check domain access settings if they are on the form.
      elseif (!empty($form[self::DOMAIN_ACCESS]) && ($form[self::DOMAIN_ACCESS]['#access'] ?? FALSE)) {
        $domain_access = $this->processDomainAccessField($form_state->getValue(self::DOMAIN_ACCESS));
      }
      // No domain access settings on the form, so we check the entity itself.
      else {
        $entity = $form_state->getFormObject()->getEntity();
        /** @var \Drupal\User\UserInterface $user */
        $user = $this->entityTypeManager->getStorage('user')->load($this->accountManager->id());
        $domain_access = $this->getEntityDomainAccessForUser($entity, $user);
      }
    }
    else {
      $domain_access = TRUE;
    }
    return $domain_access;
  }

  /**
   * Determine the domain access for an entity based on the user's permissions.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The entity for which domain access is being evaluated.
   * @param \Drupal\Core\Entity\FieldableEntityInterface $user
   *   The user whose domain access is being checked.
   *
   * @return bool|array
   *   The domain access information for the user on the entity.
   */
  public function getEntityDomainAccessForUser($entity, $user) {
    if (DomainAccessManager::getAllValue($entity)
      // If the entity doesn't have domain access fields,
      // we assume it's open to all affiliates.
      || !$this->hasDomainAccessFields($entity)) {
      if (DomainAccessManager::getAllValue($user)
        || $user->hasPermission('publish to any domain')) {
        $domain_access = TRUE;
      }
      else {
        $domain_access = DomainAccessManager::getAccessValues($user);
      }
    }
    else {
      $entity_domains = DomainAccessManager::getAccessValues($entity);
      if (DomainAccessManager::getAllValue($user)
        || $user->hasPermission('publish to any domain')) {
        $domain_access = $entity_domains;
      }
      else {
        $user_domains = DomainAccessManager::getAccessValues($user);
        $domain_access = array_intersect_key($entity_domains, $user_domains);
      }
    }
    return $domain_access;
  }

  /**
   * Determines the domain access settings for altering the entity form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return bool|array
   *   Returns TRUE if the domain access module is not installed if domain
   *   access is enabled for all domains. Returns an array of target IDs if
   *   specific domain access is configured.
   */
  public function getAlterFormDomainAccess($form, $form_state) {
    if (!$this->moduleHandler->moduleExists('domain_access')) {
      return TRUE;
    }

    $entity = $form_state->getFormObject()->getEntity();
    /** @var \Drupal\User\UserInterface $user */
    $user = $this->entityTypeManager->getStorage('user')->load($this->accountManager->id());
    // Domain access widgets are present on this form.
    if (($this->hasDomainAccessWidgets($form) && $this->hasDynamicDomainPaths($form))
      // Or form entity doesn't have domain access fields.
      || !$this->hasDomainAccessFields($entity)) {
      if (DomainAccessManager::getAllValue($user)
        || $user->hasPermission('publish to any domain')) {
        $domain_access = TRUE;
      }
      else {
        // We allow only the domains the user is assigned to.
        $domain_access = DomainAccessManager::getAccessValues($user);
      }
    }
    // No domain access fields on the form, so we can check the entity itself.
    else {
      if ($entity->isNew()) {
        // We need to know to which domain the entity will be published
        // before allowing the creation of the corresponding domain paths.
        $domain_access = FALSE;
      }
      else {
        // We keep only the domains the user has access to and that are
        // assigned to this entity. The domain assignments cannot be changed
        // as the access fields are not present on the form.
        $domain_access = $this->getEntityDomainAccessForUser($entity, $user);
      }
    }

    return $domain_access;
  }

  /**
   * Check if the form contains accessible domain access fields.
   *
   * @param array $form
   *   The form structure to check for domain access fields.
   *
   * @return bool
   *   TRUE if the form contains accessible domain access fields,
   *   FALSE otherwise.
   */
  protected function hasDomainAccessWidgets(array $form) {
    return ((!empty($form[self::DOMAIN_ACCESS]) && ($form[self::DOMAIN_ACCESS]['#access'] ?? FALSE))
      || (!empty($form[self::DOMAIN_ACCESS_ALL]) && ($form[self::DOMAIN_ACCESS_ALL]['#access'] ?? FALSE)));
  }

  /**
   * Checks if the domain paths are dynamic based on the domain access field.
   *
   * @param array $form
   *   The form array to evaluate, containing domain access widgets.
   *
   * @return bool
   *   TRUE if the domain paths are dynamic, FALSE otherwise.
   */
  protected function hasDynamicDomainPaths(array $form) {
    return (isset($form[self::DOMAIN_ACCESS]['widget']['#type'])
      && (($form[self::DOMAIN_ACCESS]['widget']['#type'] === 'checkboxes')
        || ($form[self::DOMAIN_ACCESS]['widget']['#type'] === 'radios')
        || ($form[self::DOMAIN_ACCESS]['widget']['#type'] === 'select'
          && !($form[self::DOMAIN_ACCESS]['widget']['#multiple'] ?? FALSE))));
  }

  /**
   * Checks if a user account can access a specific domain.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account to check.
   * @param int $domain_id
   *   The domain ID to check access for.
   *
   * @return bool
   *   TRUE if the user can access the domain, FALSE otherwise.
   */
  public function userCanAccessDomain(AccountInterface $account, $domain_id): bool {
    if (!$this->moduleHandler->moduleExists('domain_access')) {
      return TRUE;
    }

    // Users with global permission can access any domain.
    if ($account->hasPermission('publish to any domain')) {
      return TRUE;
    }

    // Load the full user entity to check domain assignments.
    /** @var \Drupal\user\UserInterface $user */
    $user = $this->entityTypeManager->getStorage('user')->load($account->id());
    if (!$user) {
      return FALSE;
    }

    // Check if the user has access to all domains.
    if (DomainAccessManager::getAllValue($user)) {
      return TRUE;
    }

    // Check if the user has access to this specific domain.
    $user_domains = DomainAccessManager::getAccessValues($user);
    if (isset($user_domains[$domain_id])) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * Retrieves the domains the user has access to.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account to check.
   *
   * @return array
   *   An array of domains that the user has access to, keyed by domain ID.
   */
  public function getUserAccessibleDomains(AccountInterface $account) {
    $domains = $this->entityTypeManager->getStorage('domain')->loadMultipleSorted();

    if (!$this->moduleHandler->moduleExists('domain_access')) {
      return $domains;
    }

    // Users with global permission can access any domain.
    if ($account->hasPermission('publish to any domain')) {
      return $domains;
    }

    // Load the full user entity to check domain assignments.
    if ($account instanceof UserInterface) {
      $user = $account;
    }
    else {
      /** @var \Drupal\user\UserInterface $user */
      $user = $this->entityTypeManager->getStorage('user')->load($account->id());
      if (!$user) {
        return [];
      }
    }

    // Check if the user has access to all domains.
    if (DomainAccessManager::getAllValue($user)) {
      return $domains;
    }

    $accessible_domains = [];

    // Check if the user has access to the domains.
    $user_domains = DomainAccessManager::getAccessValues($user);
    foreach ($domains as $domain_id => $domain) {
      if (array_key_exists($domain_id, $user_domains)) {
        $accessible_domains[$domain_id] = $domain;
      }
    }

    return $accessible_domains;
  }

  /**
   * Checks if the entity has domain access fields.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The entity to check for domain access fields.
   *
   * @return bool
   *   TRUE if the entity has domain access fields, FALSE otherwise.
   */
  protected function hasDomainAccessFields(FieldableEntityInterface $entity) {
    return $entity->hasField(self::DOMAIN_ACCESS) || $entity->hasField(self::DOMAIN_ACCESS_ALL);
  }

}
