<?php

namespace Drupal\domain_path\Plugin\Field\FieldType;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\path\Plugin\Field\FieldType\PathFieldItemList;

/**
 * The domain path field Item.
 */
class DomainPathFieldItemList extends PathFieldItemList {

  /**
   * {@inheritdoc}
   */
  protected function computeValue() {
    $user = \Drupal::service('current_user');
    $domains = \Drupal::service('domain_path.helper')->getUserAccessibleDomains($user);
    // For a new entity, default to creating new aliases.
    $entity = $this->getEntity();
    if ($entity->isNew()) {
      $i = 0;
      foreach ($domains as $domain_id => $domain) {
        $value = [
          'langcode' => $this->getLangcode(),
          'domain_id' => $domain_id,
        ];
        $this->list[$i] = $this->createItem($i, $value);
        $i++;
      }
    }
    else {
      /** @var \Drupal\path_alias\AliasRepositoryInterface $path_alias_repository */
      $domain_path_alias_storage = \Drupal::entityTypeManager()
        ->getStorage('domain_path');
      $paths = $domain_path_alias_storage
        ->loadByProperties([
          'source' => '/' . $entity->toUrl()->getInternalPath(),
          'language' => $this->getLangcode(),
        ]);
      $paths_by_domain = [];
      foreach ($paths as $path) {
        $paths_by_domain[$path->getDomainId()] = $path;
      }

      $i = 0;
      foreach ($domains as $domain_id => $domain) {
        if (isset($paths_by_domain[$domain_id])) {
          $path = $paths_by_domain[$domain_id];
          $value = [
            'alias' => $path->getAlias(),
            'pid' => $path->id(),
            'langcode' => $path->getLanguageCode(),
            'domain_id' => $path->getDomainId(),
          ];
        }
        else {
          $value = [
            'langcode' => $this->getLangcode(),
            'domain_id' => $domain_id,
          ];
        }
        $this->list[$i] = $this->createItem($i, $value);
        $i++;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function defaultAccess($operation = 'view', ?AccountInterface $account = NULL) {
    if ($operation == 'view') {
      return AccessResult::allowed();
    }
    return AccessResult::allowedIfHasPermissions($account, ['edit domain paths', 'administer domain paths'], 'OR')->cachePerPermissions();
  }

  /**
   * {@inheritdoc}
   */
  public function delete() {
    // Ensure that the computed value is up to date.
    $this->ensureComputedValue();

    // Delete all aliases associated with this entity.
    $entity = $this->getEntity();
    $langcode_list = [$entity->language()->getId()];

    // The default translation may also have language-neutral aliases.
    if ($entity->isDefaultTranslation()) {
      $langcode_list[] = LanguageInterface::LANGCODE_NOT_SPECIFIED;
    }

    $path_alias_storage = \Drupal::entityTypeManager()->getStorage('domain_path');
    $entities = $path_alias_storage->loadByProperties([
      'source' => '/' . $entity->toUrl()->getInternalPath(),
      'language' => $langcode_list,
    ]);
    $path_alias_storage->delete($entities);
  }

}
