<?php

namespace Drupal\domain_path\Plugin\Field\FieldWidget;

// @phpstan-ignore-next-line
use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\domain_path\DomainPathHelper;
use Drupal\path\Plugin\Field\FieldWidget\PathWidget;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'path' widget.
 *
 * @FieldWidget(
 *   id = "domain_path",
 *   label = @Translation("Domain path URL alias"),
 *   field_types = {
 *     "domain_path"
 *   }
 * )
 * @phpstan-ignore-next-line
 */
#[FieldWidget(
  id: 'domain_path',
  label: new TranslatableMarkup('Domain path URL alias'),
  field_types: ['domain_path'],
)]
class DomainPathWidget extends PathWidget {

  /**
   * The domain path helper.
   *
   * @var \Drupal\domain_path\DomainPathHelper
   */
  protected DomainPathHelper $domainPathHelper;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * The domain_path.settings config object.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->domainPathHelper = $container->get('domain_path.helper');
    $instance->currentUser = $container->get('current_user');
    $config_factory = $container->get('config.factory');
    $instance->config = $config_factory->get('domain_path.settings');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $entity = $items->getEntity();

    $element['#type'] = 'container';

    // Items that will be same for all domains.
    $element['source'] = [
      '#type' => 'value',
      '#value' => !$entity->isNew() ? '/' . $entity->toUrl()->getInternalPath() : NULL,
    ];
    $element['langcode'] = [
      '#type' => 'value',
      '#value' => $items[$delta]->langcode,
    ];
    $element['pid'] = [
      '#type' => 'value',
      '#value' => $items[$delta]->pid,
    ];
    $element['domain_id'] = [
      '#type' => 'value',
      '#value' => $items[$delta]->domain_id,
    ];
    $element['alias'] = [
      '#type' => 'textfield',
      '#title' => $element['#title'],
      '#default_value' => $items[$delta]->alias,
      '#required' => $element['#required'],
      '#maxlength' => 255,
      '#description' => $this->t('Specify an alternative path by which this data can be accessed. For example, type "/about" when writing an about page.'),
    ];

    $element += [
      '#element_validate' => [[static::class, 'validateFormElement']],
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function validateFormElement(array &$element, FormStateInterface $form_state) {
    // Trim the submitted value of whitespace and slashes.
    $alias = rtrim(trim($element['alias']['#value']), " \\/");
    if ($alias !== '') {
      $form_state->setValueForElement($element['alias'], $alias);

      /** @var \Drupal\domain_path\DomainPathInterface $path_alias */
      $path_alias = \Drupal::entityTypeManager()
        ->getStorage('domain_path')->create([
          'path' => $element['source']['#value'],
          'alias' => $alias,
          'langcode' => $element['langcode']['#value'],
          'domain_id' => $element['domain_id']['#value'],
        ]);

      $violations = $path_alias->validate();
      foreach ($violations as $violation) {
        // Newly created entities do not have a system path yet, so we need to
        // disregard some violations.
        if (!$path_alias->getSource() && $violation->getPropertyPath() === 'source') {
          continue;
        }
        $form_state->setError($element['alias'], $violation->getMessage());
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function formMultipleElements(FieldItemListInterface $items, array &$form, FormStateInterface $form_state) {
    $elements = [];
    $domains = $this->getDomains();
    $alias_title = $this->config->get('alias_title');
    foreach ($items as $delta => $item) {
      $domain_id = $items[$delta]->domain_id;
      $domain = $domains[$domain_id];
      if ($domain) {
        $label = $domain->label();
        if ($alias_title === 'hostname') {
          $label = $domain->getHostname();
        }
        elseif ($alias_title === 'url') {
          $label = $domain->getPath();
        }
      }
      else {
        $label = $domain_id;
      }
      // For multiple fields, title and description are handled by the wrapping
      // table.
      $element = [
        '#title' => $this->t('URL alias for @domain_name', [
          '@domain_name' => $label,
        ]),
        '#title_display' => 'invisible',
        '#description' => '',
      ];

      $element = $this->formSingleElement($items, $delta, $element, $form, $form_state);

      $element['#delta'] = $delta;
      $element['#weight'] = $delta;

      $elements[$domain_id] = $element;
    }

    return $elements;
  }

  /**
   * The domain list available for the current user.
   *
   * @return array
   *   The array with domains.
   */
  private function getDomains() {
    return $this->domainPathHelper->getUserAccessibleDomains($this->currentUser);
  }

}
