<?php

namespace Drupal\domain_path\Plugin\Validation\Constraint;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\domain_path\DomainAliasManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;

/**
 * Constraint validator for creating domain alias that does not already exist.
 */
class DomainPathUniqueConstraintValidator extends ConstraintValidator implements ContainerInjectionInterface {

  public function __construct(protected DomainAliasManagerInterface $domainAliasManager) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('domain_path.path_alias_manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function validate($value, Constraint $constraint) {
    $existing_source = $this->domainAliasManager->getDomainPathByAlias(
      $value->alias, $value->domain_id, $value->langcode);
    if ($existing_source && $existing_source !== $value->alias) {
      // Get the entity from the root of the validation context.
      $entity = $this->context->getRoot()->getValue();
      if ($entity->isNew()) {
        $duplicate = TRUE;
      }
      else {
        $entity_source = '/' . $entity->toUrl()->getInternalPath();
        $duplicate = $existing_source !== $entity_source;
      }
      if ($duplicate) {
        $this->context->addViolation(
          $constraint->message,
          ['%alias' => $value->alias, '%domain' => $value->domain_id],
        );
      }
    }
  }

}
