<?php

namespace Drupal\Tests\domain_path\Functional;

use Drupal\path_alias\Entity\PathAlias;

// @phpstan-ignore-next-line
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests the domain path fallback to core aliases.
 *
 * @see https://www.drupal.org/i/3566322
 *
 * @group domain_path
 *
 * @phpstan-ignore-next-line
 */
#[RunTestsInSeparateProcesses]
class DomainPathFallbackTest extends DomainPathTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'path',
    'path_alias',
  ];

  /**
   * Test domain path fallback to core aliases.
   */
  public function testDomainPathFallback() {
    // Create a node.
    $node = $this->drupalCreateNode([
      'type' => 'page',
      'title' => 'Test Node',
    ]);

    // Create a core path alias for this node.
    $core_alias = '/core-alias';
    $path_alias = PathAlias::create([
      'path' => '/node/' . $node->id(),
      'alias' => $core_alias,
      'langcode' => 'en',
    ]);
    $path_alias->save();

    // If no domain-specific alias exists, it should fall back to the core one.
    $alias_manager = $this->container->get('path_alias.manager');
    $path_from_core = $alias_manager->getPathByAlias($core_alias, 'en');
    $this->assertEquals('/node/' . $node->id(), $path_from_core, 'The core alias should be resolved when no domain alias exists.');
    // Check it a second time to ensure the cache is working properly.
    $path_from_core = $alias_manager->getPathByAlias($core_alias, 'en');
    $this->assertEquals('/node/' . $node->id(), $path_from_core, 'The core alias should be resolved when no domain alias exists.');

    // Now create a domain-specific alias and ensure it's preferred.
    $domain = reset($this->domains);
    $domain_alias = '/domain-alias';

    $domain_path_storage = $this->entityTypeManager->getStorage('domain_path');
    $domain_path_alias = $domain_path_storage->create([
      'domain_id' => $domain->id(),
      'source' => '/node/' . $node->id(),
      'alias' => $domain_alias,
      'language' => 'en',
    ]);
    $domain_path_alias->save();

    // We need to make sure we are on the right domain for the negotiator.
    // In functional tests, this can be tricky.
    // DomainPathAliasManager uses domainNegotiator->getActiveDomain().
    // For this test, we can manually set the active domain if needed,
    // or rely on how DomainTestBase sets up the request.
    $this->container->get('domain.negotiator')->setActiveDomain($domain);

    $resolved_path = $alias_manager->getPathByAlias($domain_alias, 'en');
    $this->assertEquals('/node/' . $node->id(), $resolved_path, 'The domain-specific alias should be resolved.');

    // Test that the core alias still works as a fallback even if another
    // domain has an alias.
    $path_from_core = $alias_manager->getPathByAlias($core_alias, 'en');
    $this->assertEquals('/node/' . $node->id(), $path_from_core, 'The core alias should still work as a fallback.');

    // Test getAliasByPath fallback to core alias when no domain-specific
    // alias exists for a path.
    // Create another node with only a core alias.
    $node2 = $this->drupalCreateNode([
      'type' => 'page',
      'title' => 'Test Node 2',
    ]);
    $core_alias2 = '/core-alias-2';
    $path_alias2 = PathAlias::create([
      'path' => '/node/' . $node2->id(),
      'alias' => $core_alias2,
      'langcode' => 'en',
    ]);
    $path_alias2->save();

    // With active domain, getAliasByPath should fall back to core alias.
    $resolved_alias = $alias_manager->getAliasByPath('/node/' . $node2->id(), 'en');
    $this->assertEquals($core_alias2, $resolved_alias, 'getAliasByPath should fall back to core alias.');
    // Check it a second time to ensure the cache is working properly.
    $resolved_alias = $alias_manager->getAliasByPath('/node/' . $node2->id(), 'en');
    $this->assertEquals($core_alias2, $resolved_alias, 'getAliasByPath should fall back to core alias.');

    // Now create a domain-specific alias for node2 and ensure it's preferred.
    $domain_alias2 = '/domain-alias-2';
    $domain_path_alias2 = $domain_path_storage->create([
      'domain_id' => $domain->id(),
      'source' => '/node/' . $node2->id(),
      'alias' => $domain_alias2,
      'language' => 'en',
    ]);
    $domain_path_alias2->save();

    // Clear static cache of alias manager to ensure it doesn't return the core
    // alias from cache.
    $alias_manager->cacheClear();

    $resolved_alias = $alias_manager->getAliasByPath('/node/' . $node2->id(), 'en');
    $this->assertEquals($domain_alias2, $resolved_alias, 'getAliasByPath should prefer domain-specific alias.');
  }

}
