<?php

namespace Drupal\Tests\domain_path\Functional;

use Drupal\domain_access\DomainAccessManagerInterface;
// @phpstan-ignore-next-line
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Visibility tests for Domain Path with Domain Access using autocomplete.
 *
 * Mirrors DomainPathVisibilityWithDomainAccessTest but targets a bundle where
 * `field_domain_access` is rendered as a standard entity reference
 * autocomplete (non-tags) with unlimited cardinality.
 *
 * Scenarios covered:
 * - Admin user (has "publish to any domain") sees domain_path fields for all
 *   domains on the node add form for the autocomplete bundle.
 * - Editor user (has only assigned domain access) sees domain_path fields only
 *   for their assigned domain(s); others are hidden.
 *
 * @group domain_path
 *
 * @phpstan-ignore-next-line
 */
#[RunTestsInSeparateProcesses]
class DomainPathVisibilityWithDomainAccessAutocompleteTest extends DomainPathTestBase {

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Switch the Domain Access field widget to standard entity reference
    // autocomplete. With unlimited cardinality this renders an indexed
    // [0][target_id] and an "Add another item" control.
    /** @var \Drupal\Core\Entity\Display\EntityDisplayInterface $display */
    $display = $this->entityTypeManager
      ->getStorage('entity_form_display')
      ->load('node.page.default');
    if ($display) {
      $display->setComponent('field_domain_access', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 40,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => 60,
          'placeholder' => '',
        ],
      ])->save();
    }

    // Ensure that for the standard 'page' bundle, the Domain Access field
    // defaults to both created domains. This helps tests that expect
    // multi-valued defaults on the node add form.
    $domain_ids = array_keys($this->domains);
    /** @var \Drupal\field\FieldConfigInterface|null $field_domain_access */
    $field_domain_access = $this->entityTypeManager
      ->getStorage('field_config')
      ->load('node.page.field_domain_access');
    if ($field_domain_access) {
      $default = [];
      foreach ($domain_ids as $id) {
        $default[] = ['target_id' => $id];
      }
      $field_domain_access->setDefaultValue($default)->save();
    }
  }

  /**
   * Asserts visibility of domain path fields for admin and editor on add form.
   */
  public function testDomainPathFieldVisibilityWithAutocomplete(): void {
    // Prepare two domain IDs from base setup.
    $domain_ids = array_keys($this->domains);
    $domain_a = $domain_ids[0];
    $domain_b = $domain_ids[1];

    // Create Admin with global publish permission.
    $admin = $this->drupalCreateUser([
      'access content',
      'create page content',
      'edit any page content',
      'view domain entity',
      'publish to any domain',
      'administer domain paths',
    ]);

    // Admin should see the autocomplete widget and all domain_path fields.
    $this->drupalLogin($admin);
    $this->drupalGet('node/add/page');
    $this->assertSession()->statusCodeEquals(200);
    // Verify the Domain Access widget is the standard autocomplete input.
    $this->assertSession()->fieldExists('field_domain_access[0][target_id]');
    // We cannot see the domain_path fields for all domains yet.
    foreach ($this->domains as $domain) {
      $this->assertSession()->fieldNotExists('domain_path[' . $domain->id() . '][alias]');
    }

    $edit = [];
    $edit['title[0][value]'] = $this->randomMachineName(8);
    $edit['body[0][value]'] = $this->randomMachineName(16);
    $this->submitForm($edit, t('Save'));

    $this->drupalGet('node/1/edit');
    // The path fields for both domains should now be visible.
    foreach ($this->domains as $domain) {
      $this->assertSession()->fieldExists('domain_path[' . $domain->id() . '][alias]');
    }

    // Create Editor limited to a single domain (no global publish permission).
    $editor = $this->drupalCreateUser([
      'access content',
      'create page content',
      'edit own page content',
      'view domain entity',
      'publish to any assigned domain',
      'edit domain paths',
    ]);
    // Assign editor to domain A via user field.
    $this->addDomainsToEntity('user', $editor->id(), [$domain_a], DomainAccessManagerInterface::DOMAIN_ACCESS_FIELD);

    // Login as editor and open the add form for the autocomplete bundle.
    $this->drupalLogin($editor);
    $this->drupalGet('node/add/page');
    $this->assertSession()->statusCodeEquals(200);
    // Editor should still have the autocomplete widget available.
    $this->assertSession()->fieldExists('field_domain_access[0][target_id]');

    $edit = [];
    $edit['title[0][value]'] = $this->randomMachineName(8);
    $edit['body[0][value]'] = $this->randomMachineName(16);
    $this->submitForm($edit, t('Save'));

    $this->drupalGet('node/2/edit');
    // The form alteration should expose domain_path fields only for domains the
    // user can publish to (domain_a) and hide others (domain_b).
    $this->assertSession()->fieldExists('domain_path[' . $domain_a . '][alias]');
    $this->assertSession()->fieldNotExists('domain_path[' . $domain_b . '][alias]');
  }

}
