<?php

namespace Drupal\domain_theme_switch\Form;

use Drupal\Core\Config\CachedStorage;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\domain_config\Config\DomainConfigCollectionUtils;
use Drupal\domain_config_ui\DomainConfigUIManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The Domain Theme Switch module configuration form.
 */
class DomainThemeSwitchConfigForm extends FormBase {

  /**
   * Construct function.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory load.
   * @param \Drupal\Core\Config\CachedStorage $configStorage
   *   Config storage.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Extension\ThemeHandlerInterface $themeHandler
   *   The theme handler.
   * @param \Drupal\domain_config_ui\DomainConfigUIManager $domainConfigUiManager
   *   The domain config ui manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    protected CachedStorage $configStorage,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ThemeHandlerInterface $themeHandler,
    protected DomainConfigUIManager $domainConfigUiManager,
  ) {
    $this->setConfigFactory($config_factory);
  }

  /**
   * Create function return static domain loader configuration.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   Load the ContainerInterface.
   *
   * @return static
   *   return domain loader configuration.
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.storage'),
      $container->get('entity_type.manager'),
      $container->get('theme_handler'),
      $container->get('domain_config_ui.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'domain_theme_switch.settings',
    ];
  }

  /**
   * Form ID is domain_theme_switch_config_form.
   *
   * @return string
   *   Return form ID.
   */
  public function getFormId() {
    return 'domain_theme_switch_config_form';
  }

  /**
   * Function to get the list of installed themes.
   *
   * @return array
   *   The complete theme registry data array.
   */
  public function getThemeList() {
    $themeName = [];
    foreach ($this->themeHandler->listInfo() as $key => $value) {
      $themeName[$key] = $value->info['name'];
    }
    return $themeName;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $base_system_theme = $this->configStorage->read('system.theme');
    $default_site_theme = $base_system_theme['default'];
    $default_admin_theme = $base_system_theme['admin'];

    $themeNames = $this->getThemeList();
    $domains = $this->entityTypeManager->getStorage('domain')->loadMultipleSorted();
    foreach ($domains as $domain) {
      $domain_id = $domain->id();
      $override = $this->domainConfigUiManager->isConfigurationRegisteredForDomain($domain_id, 'system.theme');
      $domain_collection = $this->configStorage->createCollection(
        DomainConfigCollectionUtils::createDomainConfigCollectionName($domain_id));
      $system_theme_config = $domain_collection->read('system.theme');
      if ($system_theme_config) {
        $override_site_theme = $system_theme_config['default'] ?? NULL;
        $override_admin_theme = $system_theme_config['admin'] ?? NULL;
      }
      else {
        $override_site_theme = NULL;
        $override_admin_theme = NULL;
      }

      $form[$domain_id] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Select Theme for "@domain"', ['@domain' => $domain->get('name')]),
      ];
      $form[$domain_id][$domain_id . '_override'] = [
        '#title' => $this->t('Enable theme override'),
        '#type' => 'checkbox',
        '#default_value' => $override,
      ];
      $form[$domain_id][$domain_id . '_site'] = [
        '#title' => $this->t('Site theme for domain'),
        '#type' => 'select',
        '#options' => $themeNames,
        '#default_value' => $override_site_theme ?? $default_site_theme,
        '#states' => [
          'visible' => [
            ':input[name="' . $domain_id . '_override"]' => ['checked' => TRUE],
          ],
        ],
      ];
      $form[$domain_id][$domain_id . '_admin'] = [
        '#title' => $this->t('Admin theme for domain'),
        '#type' => 'select',
        '#options' => $themeNames,
        '#default_value' => $override_admin_theme ?? $default_admin_theme,
        '#states' => [
          'visible' => [
            ':input[name="' . $domain_id . '_override"]' => ['checked' => TRUE],
          ],
        ],
      ];
    }

    if (count($domains) === 0) {
      $form['domain_theme_switch_message'] = [
        '#markup' => $this->t('Zero domain records found. Please @link to create the domain.', [
          '@link' => Link::fromTextAndUrl($this->t('click here'), Url::fromRoute('domain.admin'))->toString(),
        ]),
      ];
    }
    else {
      $form['actions']['#type'] = 'actions';
      $form['actions']['submit'] = [
        '#type' => 'submit',
        '#value' => $this->t('Save configuration'),
        '#button_type' => 'primary',
      ];
    }

    // By default, render the form using system-config-form.html.twig.
    $form['#theme'] = 'system_config_form';

    return $form;
  }

  /**
   * Validate function for the form.
   *
   * @param array $form
   *   Form items.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state to validate.
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $domains = $this->entityTypeManager->getStorage('domain')->loadMultiple();
    foreach ($domains as $domain) {
      $domain_id = $domain->id();
      $original_override =
        $this->domainConfigUiManager->isConfigurationRegisteredForDomain(
          $domain_id, 'system.theme');
      $new_override = $form_state->getValue($domain_id . '_override');
      if ($original_override && !$new_override) {
        $this->domainConfigUiManager->deleteConfigurationOverridesForDomain($domain_id, 'system.theme');
      }
      elseif ($new_override) {
        $update = FALSE;
        $domain_collection =
          $this->configStorage->createCollection(
            DomainConfigCollectionUtils::createDomainConfigCollectionName($domain_id));
        $system_theme_config = $domain_collection->read('system.theme');
        $original_site = $system_theme_config['default'];
        $original_admin_theme = $system_theme_config['admin'];
        $new_site = $form_state->getValue($domain_id . '_site');
        $new_admin = $form_state->getValue($domain_id . '_admin');
        if (!$original_override) {
          $this->domainConfigUiManager->addConfigurationsToDomain($domain_id, ['system.theme']);
          $update = TRUE;
        }
        elseif ($original_site !== $new_site || $original_admin_theme !== $new_admin) {
          $update = TRUE;
        }
        if ($update) {
          $system_theme_config['default'] = $new_site;
          $system_theme_config['admin'] = $new_admin;
          $domain_collection->write('system.theme', $system_theme_config);
        }
      }
    }
  }

}
