<?php

namespace Drupal\draggable_dashboard\Form;

use Drupal\Core\Form\FormStateInterface;
use Drupal\draggable_dashboard\Entity\DashboardEntityInterface;

/**
 * Class DashboardAssignBlockForm.
 *
 * @package Drupal\draggable_dashboard\Form
 */
class DashboardAssignBlockForm extends DashboardBlockFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'draggable_dashboard_assign_block';
  }

  /**
   * Initialize the form state and the entity before the first form build.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state object.
   * @param \Drupal\draggable_dashboard\Entity\DashboardEntityInterface $dashboard_entity
   *   Dashboard object.
   * @param string $plugin_id
   *   Plugin id of the block used.
   */
  protected function init(FormStateInterface $form_state, DashboardEntityInterface $dashboard_entity, $plugin_id = '') {
    parent::init($form_state, $dashboard_entity);
    $this->block = ['settings' => ['id' => $plugin_id]];
  }

  /**
   * Builds assign block form.
   *
   * @param array $form
   *   Form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   * @param \Drupal\draggable_dashboard\Entity\DashboardEntityInterface|null $dashboard_entity
   *   Current dashboard.
   * @param string $plugin_id
   *   Block plugin id.
   * @param int $region
   *   Column to insert block to.
   *
   * @return array
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?DashboardEntityInterface $dashboard_entity = NULL, $plugin_id = '', $region = 1) {
    // During the initial form build, add this form object to the form state and
    // allow for initial preparation before form building and processing.
    if (!$form_state->has('form_initialized')) {
      $this->init($form_state, $dashboard_entity, $plugin_id);
    }
    $regions = [];
    for ($i = 1; $i <= $dashboard_entity->get('columns'); $i++) {
      $regions[$i] = t('Column') . ' ' . $i;
    }
    $form['column'] = [
      '#type' => 'select',
      '#title' => $this->t('Region'),
      '#description' => $this->t('Select the region where this block should be displayed.'),
      '#default_value' => $region,
      '#required' => TRUE,
      '#options' => $regions,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * @param $machine_name
   * @param $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *
   * @return bool
   */
  public function blockIdExists($machine_name, $form, FormStateInterface $form_state) {
    // Get existing blocks from current dashboard.
    $dashboard_blocks = $this->dashboard->get('blocks');
    // If empty return early.
    if (empty($dashboard_blocks)) {
      return FALSE;
    }
    $blocks = array_keys($dashboard_blocks);
    return in_array($machine_name, $blocks, TRUE);
  }

  /**
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);
    $settings = $form_state->getValue('settings');
    $id = $form_state->getValue('id');
    $region = $form_state->getValue('column');
    $blocks = $this->dashboard->get('blocks') ? $this->dashboard->get('blocks') : [];
    $blocks[$id] = [
      'column' => (int) $region,
      'settings' => $settings,
      'weight' => 0,
    ];

    // Save relation.
    $this->dashboard->set('blocks', $blocks)->save();
    // Redirect to manage blocks screen.
    $form_state->setRedirect('entity.dashboard_entity.edit_form', ['dashboard_entity' => $this->dashboard->id()]);
  }

}
