<?php

namespace Drupal\draggable_dashboard\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\draggable_dashboard\Entity\DashboardEntityInterface;

/**
 * Class DashboardDeleteBlockForm.
 *
 * @package Drupal\draggable_dashboard\Form
 */
class DashboardDeleteBlockForm extends ConfirmFormBase {

  /**
   * Current dashboard.
   *
   * @var \Drupal\draggable_dashboard\Entity\DashboardEntityInterface
   */
  protected $dashboard;

  /**
   * Block to delete.
   *
   * @var array
   */
  protected $block = [];

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want delete block `%title`?', ['%title' => $this->block['settings']['label']]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return Url::fromRoute('entity.dashboard_entity.edit_form', ['dashboard_entity' => $this->dashboard->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'draggable_dashboard_block_delete';
  }

  /**
   * Initialize the form state and the entity before the first form build.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state object.
   * @param \Drupal\draggable_dashboard\Entity\DashboardEntityInterface $dashboard_entity
   *   Dashboard object.
   * @param string $block_id
   *   ID of the block to delete.
   */
  protected function init(FormStateInterface $form_state, DashboardEntityInterface $dashboard_entity, $block_id) {
    // Flag that this form has been initialized.
    $form_state->set('form_initialized', TRUE);
    $this->dashboard = $dashboard_entity;
    $blocks = $dashboard_entity->get('blocks');
    if (!empty($blocks[$block_id])) {
      $this->block = $blocks[$block_id];
    }
  }

  /**
   * @param array $form
   *   Form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state object.
   * @param \Drupal\draggable_dashboard\Entity\DashboardEntityInterface $dashboard_entity
   *   Dashboard object.
   * @param string $block_id
   *   ID of the block to delete.
   *
   * @return array
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?DashboardEntityInterface $dashboard_entity = NULL, $block_id = '') {
    // During the initial form build, add this form object to the form state and
    // allow for initial preparation before form building and processing.
    if (!$form_state->has('form_initialized')) {
      $this->init($form_state, $dashboard_entity, $block_id);
    }
    $form['block_id'] = [
      '#type' => 'value',
      '#value' => $block_id,
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $block_id = $form_state->getValue('block_id');
    $blocks = $this->dashboard->get('blocks');
    if (isset($blocks[$block_id])) {
      unset($blocks[$block_id]);
    }
    $this->dashboard->set('blocks', $blocks)->save();
    $form_state->setRedirect('entity.dashboard_entity.edit_form', ['dashboard_entity' => $this->dashboard->id()]);
  }

}
