<?php

namespace Drupal\draggable_dashboard\Plugin\Block;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Block\BlockManagerInterface;
use Drupal\Core\Block\TitleBlockPluginInterface;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Controller\TitleResolverInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a draggable block with a simple text.
 *
 * @Block(
 *   id = "draggable_dashboard_block",
 *   admin_label = @Translation("Draggable Block"),
 *   deriver = "Drupal\draggable_dashboard\Plugin\Block\DraggableBlockDeriver"
 * )
 */
class DraggableBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * Entity Type Manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The title resolver.
   *
   * @var \Drupal\Core\Controller\TitleResolverInterface
   */
  protected $titleResolver;

  /**
   * Block manager service.
   *
   * @var \Drupal\Core\Block\BlockManagerInterface
   */
  protected $blockManager;

  /**
   * Current user service.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * DraggableBlock constructor.
   *
   * @param array $configuration
   *   Plugin configuration.
   * @param string $plugin_id
   *   Plugin id.
   * @param array $plugin_definition
   *   Plugin definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   Entity Type Manager.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   Request stack.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   Current Route Matcher.
   * @param \Drupal\Core\Theme\ThemeManagerInterface $theme_manager
   *   The theme manager.
   * @param \Drupal\Core\Controller\TitleResolverInterface $title_resolver
   *   Title resolver.
   * @param \Drupal\Core\Block\BlockManagerInterface $block_plugin_manager
   *   Block manager.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   Current user.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function __construct(array $configuration, $plugin_id, array $plugin_definition, EntityTypeManagerInterface $entity_type_manager, RequestStack $request_stack, RouteMatchInterface $route_match, TitleResolverInterface $title_resolver, BlockManagerInterface $block_plugin_manager, AccountProxyInterface $current_user) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->titleResolver = $title_resolver;
    $this->entityTypeManager = $entity_type_manager;
    $this->blockManager = $block_plugin_manager;
    $this->currentUser = $current_user;
    $this->routeMatch = $route_match;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('request_stack'),
      $container->get('current_route_match'),
      $container->get('title_resolver'),
      $container->get('plugin.manager.block'),
      $container->get('current_user')
    );
  }

  /**
   * Gets the current dashboard entity.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   */
  private function getDashboard() {
    $dashboard_id = str_replace('draggable_dashboard_block:draggable_dashboard_', '', $this->getConfiguration()['id']);
    return $this->entityTypeManager->getStorage('dashboard_entity')->load($dashboard_id);
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $dashboard = $this->getDashboard();
    $columns = [];
    if (!empty($dashboard)) {
      $all_blocks = $dashboard->get('blocks');
      $max_blocks = 0;
      // Create dashboard columns.
      for ($i = 1; $i <= $dashboard->get('columns'); $i++) {
        $blocks = [];
        if (!empty($all_blocks)) {
          foreach ($all_blocks as $key => $relation) {
            if ($relation['column'] == $i) {
              $blocks[] = $relation + ['block_id' => $key];
            }
          }
        }
        if (!empty($blocks)) {
          if ($max_blocks < count($blocks)) {
            $max_blocks = count($blocks);
          }
          foreach ($blocks as $delta => $relation) {
            $block_instance = $this->blockManager->createInstance($relation['settings']['id'], $relation['settings']);

            if ($block_instance instanceof TitleBlockPluginInterface) {
              $pageTitle = $this->titleResolver->getTitle($this->requestStack->getCurrentRequest(), $this->routeMatch->getRouteObject());
              if ($pageTitle) {
                $block_instance->setTitle($pageTitle);
              }
            }

            // Some blocks might implement access check.
            // Return empty render array if user doesn't have access.
            if ($block_instance->access($this->currentUser)) {
              // See \Drupal\block\BlockViewBuilder::buildPreRenderableBlock
              // See template_preprocess_block()
              $element = [
                '#theme' => 'block',
                '#attributes' => [
                  'id' => "block-" . $relation['block_id'],
                  'data-id' => $relation['block_id'],
                ],
                '#configuration' => $block_instance->getConfiguration(),
                '#plugin_id' => $block_instance->getPluginId(),
                '#base_plugin_id' => $block_instance->getBaseId(),
                '#derivative_plugin_id' => $block_instance->getDerivativeId(),
                '#draggable_dashboard' => TRUE,
                'content' => $block_instance->build(),
                'title' => ['#markup' => $block_instance->label()],
                'id' => ['#markup' => $relation['block_id']],
              ];

              CacheableMetadata::createFromRenderArray($element)
                ->merge(CacheableMetadata::createFromRenderArray($element['content']))
                ->addCacheableDependency($block_instance)
                ->applyTo($element);

              $columns[$i][] = $element;
            }
          }
        }
      }
    }

    return [
      '#theme' => 'draggable_dashboard_block',
      '#attributes' => [
        'data-id' => $dashboard->id(),
      ],
      '#dashboard' => $dashboard,
      '#columns' => $columns,
      '#attached' => [
        'library' => [
          'draggable_dashboard/frontend',
          'draggable_dashboard/draggable',
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  protected function blockAccess(AccountInterface $account) {
    return AccessResult::allowedIfHasPermission($account, 'access content');
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {

  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags() {
    $dashboard = $this->getDashboard();
    return $dashboard->getCacheTags();
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts() {
    return ['user.permissions'];
  }

}
