/**
 * @file
 * JavaScript behaviors for the Draggable Mapper Entity.
 */
(function ($, Drupal, drupalSettings, once) {
  /**
   * Gets the delta (index) from a paragraph item.
   */
  function getDeltaFromParagraph($paragraphItem) {
    // Check for any element with field-dme-marker-X in its ID
    const $anyElement = $paragraphItem
      .find('[id*="field-dme-marker-"]')
      .first();

    if ($anyElement.length) {
      const attrValue =
        $anyElement.attr('id') || $anyElement.attr('data-drupal-selector');
      const elemMatches = attrValue.match(/field-dme-marker-(\d+)/);

      if (elemMatches && elemMatches[1]) {
        return parseInt(elemMatches[1], 10);
      }
    }

    return 0;
  }

  /**
   * Initialize font size for a marker based on its dimensions
   */
  function resizeFont($marker) {
    // Update marker font size in real-time
    const width = $marker.width();
    const height = $marker.height();

    // Base calculation on width for tall markers
    let fontSize;
    const aspectRatio = width / height;
    if (aspectRatio < 2.5) {
      fontSize = width * 0.1;
    } else {
      const smallestDimension = Math.min(width, height);
      fontSize = smallestDimension * 0.25;
    }

    fontSize = Math.max(fontSize, 12);
    $marker[0].style.fontSize = `${fontSize}px`;
    $($marker).attr('data-font-size', fontSize);
  }

  /**
   * Make a marker resizable with appropriate options based on marker type
   * @param {Object} $marker - jQuery object for the marker element
   */
  function makeMarkerResizable($marker) {
    // Wait until jQuery UI's resizable method is available
    if (typeof $.fn.resizable !== 'function') {
      // Retry after a short delay
      setTimeout(function () {
        makeMarkerResizable($marker);
      }, 100);
      return;
    }
    // If marker is already resizable, destroy it first
    if ($marker.hasClass('ui-resizable')) {
      $marker.resizable('destroy');
    }
    $marker.resizable({
      // aspectRatio: true,
      aspectRatio: !!$marker.hasClass('dme-marker-icon'),
      handles: 'all',
      minWidth: 100,
      minHeight: $marker.hasClass('dme-marker-icon') ? '' : 50,
      containment: '.dme-container-wrapper',
      start(event, ui) {
        $(this).addClass('dme-marker-resizing');
      },
      resize(event, ui) {
        // Update marker font size in real-time
        if ($(this).hasClass('has-title')) {
          resizeFont($(this));
        }
      },
      stop(event, ui) {
        $(this).removeClass('dme-marker-resizing');

        // Get marker data
        const markerId = $(this).attr('id');
        const delta = markerId.replace('dme-marker-', '');

        // Update hidden fields with new size information
        if (delta) {
          // Calculate size as percentage of container
          const containerWidth = $('.dme-container-wrapper').width();
          const containerHeight = $('.dme-container-wrapper').height();
          const markerWidth = ui.size.width;
          const markerHeight = ui.size.height;

          // Convert width to decimal between 0 and 1
          const widthDecimal = (markerWidth / containerWidth).toFixed(6);

          // Convert height to decimal between 0 and 1
          const heightDecimal = (markerHeight / containerHeight).toFixed(6);

          // Find and update the corresponding width input field
          const widthField = document.querySelector(
            `input[name*="field_dme_marker_width"][name*="[${delta}]"]`,
          );
          if (widthField) {
            widthField.value = widthDecimal;
          }

          // Find and update the corresponding height input field
          const heightField = document.querySelector(
            `input[name*="field_dme_marker_height"][name*="[${delta}]"]`,
          );
          if (heightField) {
            heightField.value = heightDecimal;
          }
          // Store size information in custom data attributes for persistence
          $(this).attr('data-size-width', markerWidth);
          $(this).attr('data-size-height', markerHeight);
        }
      },
    });
  }

  /**
   * Check if the unmapped wrapper is empty and show the "no markers" message with fade-in effect if it is
   */
  function checkForEmptyUnmappedWrapper() {
    const $unmappedContainer = $('.dme-unmapped-wrapper');
    // If there are no more markers in the unmapped wrapper
    if ($unmappedContainer.find('.dme-marker').length === 0) {
      // If the message doesn't exist, add it with fade in effect
      if ($unmappedContainer.find('.dme-no-markers-message').length === 0) {
        const $message = $(
          `<div class="dme-no-markers-message" style="display: none;">${Drupal.t(
            'Add new markers to be mapped',
          )}</div>`,
        );
        $unmappedContainer.append($message);
        $message.fadeIn(400);
      }
    }
  }

  /**
   * Initialize draggable functionality for marker elements
   */
  function initDraggableMarkers(context) {
    // Wait until jQuery UI's draggable method is available
    if (typeof $.fn.draggable !== 'function') {
      // Retry after a short delay
      setTimeout(function () {
        initDraggableMarkers(context);
      }, 100);
      return;
    }
    // Get all markers and make them draggable
    once('draggable', '.dme-marker', context).forEach(function (marker) {
      const $marker = $(marker);
      // Fade in the marker with animation
      $marker.addClass('visible');
      $(marker).draggable({
        // Create a clone for dragging to make transitions smoother
        helper() {
          // Create a clone that maintains fixed dimensions
          const clone = this.cloneNode(true);
          const $clone = $(clone);
          const currentWidth = $(this).outerWidth();
          const currentHeight = $(this).outerHeight();

          // Force the helper to maintain the current marker dimensions
          $clone[0].style.width = `${currentWidth}px`;
          $clone[0].style.height = `${currentHeight}px`;
          $clone[0].style.boxSizing = 'border-box';

          return $clone;
        },
        appendTo: 'body', // Attach the helper to the body to avoid containment issues during drag
        zIndex: 1000, // Ensure the dragged item appears above other elements
        opacity: 0.7, // Slightly transparent while dragging
        cursor: 'move',

        // When drag starts
        start(event, ui) {
          // Store original position for revert if needed
          $(this).data('originalPosition', $(this).position());
          $(this).data('originalParent', $(this).parent());

          // Hide the original element while dragging the clone
          this.style.opacity = '0.3';
        },

        // During drag
        drag(event, ui) {
          // Check if we're over the target container
          const $container = $('.dme-container-wrapper');
          const containerOffset = $container.offset();

          if (
            containerOffset &&
            ui.position.left >= containerOffset.left &&
            ui.position.right <= containerOffset.left + $container.width() &&
            ui.position.top >= containerOffset.top &&
            ui.position.bottom <= containerOffset.top + $container.height()
          ) {
            // Add a visual indicator that we're over the drop target
            $container.addClass('dme-drop-hover');
          } else {
            $container.removeClass('dme-drop-hover');
          }
        },

        // When drag stops
        stop(event, ui) {
          // Get marker ID from the element
          const markerId = $(this).attr('id');
          const delta = parseInt(markerId.replace('dme-marker-', ''), 10);
          const $container = $('.dme-container-wrapper');
          const $marker = $(this);

          // Reset opacity of original
          $marker[0].style.opacity = '1';

          // Check if we've dropped on the target container
          const containerOffset = $container.offset();
          const markerWidth = $(ui.helper).outerWidth();
          const markerHeight = $(ui.helper).outerHeight();
          const safetyMargin = 2; // 1px safety margin.

          // Calculate right and bottom edges
          // Use ui.offset and include the safety margin on right and bottom.
          if (
            containerOffset &&
            ui.offset.left >= containerOffset.left &&
            ui.offset.left + markerWidth <=
              containerOffset.left + $container.width() + safetyMargin &&
            ui.offset.top >= containerOffset.top &&
            ui.offset.top + markerHeight <=
              containerOffset.top + $container.height() + safetyMargin
          ) {
            // Calculate position within the target container
            const relativeX = ui.offset.left - containerOffset.left;
            const relativeY = ui.offset.top - containerOffset.top;

            // Move the original marker to the container at the right position
            $marker.detach().appendTo($container);
            const markerStyle = $marker[0].style;
            markerStyle.position = 'absolute';
            markerStyle.left = `${relativeX}px`;
            markerStyle.top = `${relativeY}px`;

            // Calculate position as percentage of container size for responsive behavior
            const containerWidth = $container.width();
            const containerHeight = $container.height();

            const posX = (relativeX / containerWidth).toFixed(6);
            const posY = (relativeY / containerHeight).toFixed(6);

            // Update the hidden form fields with the new coordinates
            let $paragraphItem = $(
              `.paragraph-type--dme-marker[data-delta="${
                delta
              }"], .paragraphs-subform[data-delta="${delta}"]`,
            );
            if (!$paragraphItem.length) {
              // Try alternative selectors as fallback
              $paragraphItem = $(`#${delta}`).closest(
                '.paragraph-type--dme-marker, .paragraphs-subform',
              );
              if (!$paragraphItem.length) {
                $paragraphItem = $(
                  '.paragraph-type--dme-marker, .paragraphs-subform',
                ).filter(function () {
                  return getDeltaFromParagraph($(this)) === delta;
                });
              }
            }

            if ($paragraphItem.length) {
              const $xField = $paragraphItem.find(
                'input[name*="field_dme_marker_x"]',
              );
              const $yField = $paragraphItem.find(
                'input[name*="field_dme_marker_y"]',
              );

              if ($xField.length && $yField.length) {
                // Update the form field values with the new coordinates
                $xField[0].value = posX;
                $yField[0].value = posY;
                // Trigger change event to ensure Drupal recognizes the change
                $xField.trigger('change');
                $yField.trigger('change');
              }
            }

            // Mark as mapped
            $marker
              .removeClass('dme-unmapped-marker')
              .addClass('dme-mapped-marker');

            // Apply resizable behavior immediately after drop
            makeMarkerResizable($marker);

            // Check if this was the last marker in the unmapped wrapper
            checkForEmptyUnmappedWrapper();
          }
          // Remove any hover effects
          $container.removeClass('dme-drop-hover');
        },
      });
    });
  }

  /**
   * Check if there are markers in the unmapped wrapper and hide the no markers message if needed
   */
  function checkAndHideNoMarkersMessage() {
    const $container = $('.dme-unmapped-wrapper');
    const $noMessage = $container.find('.dme-no-markers-message');
    const $message = $(
      `<div class="dme-no-markers-message" style="display: none;">${Drupal.t(
        'Add new markers to be mapped',
      )}</div>`,
    );
    if ($container.find('.dme-marker').length > 0) {
      // Find the message if it exists
      if ($noMessage.length > 0) {
        // Fade out the message before removing it
        $noMessage.fadeOut(400, function () {
          $(this).remove();
        });
      }
    } else if ($container.find('.dme-no-markers-message').length === 0) {
      // If no markers are present and the message doesn't exist,
      // add it with fade in.
      $container.append($message);
      $message.fadeIn(400);
    }
  }

  /**
   * Ensures a marker with the given delta exists in the map container.
   */
  function ensureMarkerExists(delta, title) {
    const $container = $('.dme-unmapped-wrapper');
    const $marker = $(`#dme-marker-${delta}`);
    if (!$marker.length) {
      // Create new marker
      let markerHtml = `<div id="dme-marker-${
        delta
      }" class="dme-marker dme-unmapped-marker js-form-wrapper form-wrapper" role="application" aria-label="Interactive component positioning interface" aria-describedby="dme-instructions" aria-live="polite">`;
      markerHtml += `<div class="dme-marker-wrapper">${title}</div></div>`;
      $container.append(markerHtml);
      // Check if we need to hide the no markers message
      checkAndHideNoMarkersMessage();
    }
    // Update marker text
    const wrapper = $marker[0]?.querySelector('.dme-marker-wrapper');
    if (wrapper) {
      wrapper.textContent = title;
    }
  }

  /**
   * Checks for existing uploaded marker icons on page load and updates markers accordingly
   */
  function initializeExistingMarkerIcons() {
    // Find all marker paragraphs with uploaded icons
    $(
      '.paragraph-type--dme-marker .field--name-field-dme-marker-icon .file, .paragraphs-subform .field--name-field-dme-marker-icon .file',
    ).each(function () {
      // Get the paragraph delta
      let $paragraphItem = $(this).closest('.paragraph-type--dme-marker');
      if (!$paragraphItem.length) {
        $paragraphItem = $(this).closest('.paragraphs-subform');
      }

      if (!$paragraphItem.length) {
        return;
      }

      const delta = getDeltaFromParagraph($paragraphItem);
      if (delta === null) {
        return;
      }

      // Find the image source - look for img or a.href for file URLs
      let imgSrc = $(this).find('img').attr('src');
      if (!imgSrc) {
        // Try finding href in case it's a file link
        imgSrc = $(this).find('a').attr('href');
      }

      if (!imgSrc) {
        return;
      }

      // Update marker with this icon
      const $marker = $(`#dme-marker-${delta}`);
      if ($marker.length) {
        // First empty the wrapper completely
        const $wrapper = $marker.find('.dme-marker-wrapper');
        $wrapper.empty();
        // Then add only the image
        $wrapper.html(`<img src="${imgSrc}" alt="Marker Icon" />`);
        // Add a class to indicate this marker has an icon
        $marker.addClass('dme-marker-icon').removeClass('has-title');
      } else {
        // We need to create the marker first
        ensureMarkerExists(delta, '');
        // Now update it with the icon
        const $newMarker = $(`#dme-marker-${delta}`);
        $newMarker
          .find('.dme-marker-wrapper')
          .empty()
          .html(`<img src="${imgSrc}" alt="Marker Icon" />`);
        $newMarker.addClass('dme-marker-icon').removeClass('has-title');
      }
    });

    // Check and hide the no markers message if needed
    checkAndHideNoMarkersMessage();
  }

  /**
   * Updates a marker based on its title input field.
   */
  function updateMarker($titleInput) {
    // Find the paragraph item containing this title field
    let $paragraphItem = $titleInput.closest('.paragraph-type--dme-marker');
    if (!$paragraphItem.length) {
      $paragraphItem = $titleInput.closest('.paragraphs-subform');
    }

    if ($paragraphItem.length) {
      // Get paragraph delta (index)
      const delta = getDeltaFromParagraph($paragraphItem);
      // Check if this paragraph already has an icon uploaded or if marker already has an icon
      const hasUploadedFile =
        $paragraphItem.find('.field--name-field-dme-marker-icon .file').length >
        0;
      const $marker = $(`#dme-marker-${delta}`);
      const hasIconClass =
        $marker.length && $marker.hasClass('dme-marker-icon');

      // If there's an uploaded file or the marker already has an icon class, don't update with the title
      if (hasUploadedFile || hasIconClass) {
        return;
      }
      // Get the input value, but use a default for the marker if empty
      const inputValue = $titleInput.length
        ? $titleInput[0].value
        : `Marker ${delta}`;
      const displayTitle = inputValue || Drupal.t('Untitled marker');

      // Create or update the marker, but only if there's no icon
      if (!$marker.length) {
        ensureMarkerExists(delta, displayTitle);
        // Mark it explicitly as a title marker
        $(`#dme-marker-${delta}`).addClass('has-title');
      } else if (!$marker.hasClass('dme-marker-icon')) {
        // Update the title if it's not an icon marker
        $marker
          .find('.dme-marker-wrapper')
          .html(`<span class="marker-text">${displayTitle}</span>`);
        $marker.addClass('has-title');
      }
    }
  }

  /**
   * Updates a marker icon based on the file input.
   */
  function updateMarkerIcon($iconInput) {
    // Find the paragraph item containing this icon field
    let $paragraphItem = $iconInput.closest('.paragraph-type--dme-marker');
    if (!$paragraphItem.length) {
      $paragraphItem = $iconInput.closest('.paragraphs-subform');
    }

    if (!$paragraphItem.length) {
      return;
    }

    // Get paragraph delta (index)
    const delta = getDeltaFromParagraph($paragraphItem);
    if (delta === null) {
      return;
    }

    const reader = new FileReader();

    // Update marker with image
    reader.onload = function (e) {
      // Update marker with image
      const $marker = $(`#dme-marker-${delta}`);
      if ($marker.length) {
        // First empty the wrapper completely
        const $wrapper = $marker.find('.dme-marker-wrapper');
        $wrapper.empty();
        // Then add only the image
        $wrapper.html(`<img src="${e.target.result}" alt="Marker Icon" />`);
        // Add a class to indicate this marker has an icon
        $marker.addClass('dme-marker-icon').removeClass('has-title');
        // Set width/height for the marker
        $marker[0].style.width = '100px';
        $marker[0].style.height = 'auto';
        if ($marker.hasClass('dme-mapped-marker')) {
          $marker.resizable({
            aspectRatio: true,
            handles: 'all',
            minWidth: 100,
            containment: '.dme-container-wrapper',
          });
        }
      } else {
        // We need to create the marker first
        ensureMarkerExists(delta, '');
        // Now update it with the icon
        const $newMarker = $(`#dme-marker-${delta}`);
        $newMarker
          .find('.dme-marker-wrapper')
          .empty()
          .html(`<img src="${e.target.result}" alt="Marker Icon" />`);
        $newMarker.addClass('dme-marker-icon').removeClass('has-title');
      }
    };
    reader.readAsDataURL($iconInput[0].files[0]);
  }

  /**
   * Check for and remove markers that no longer have a corresponding paragraph
   */
  function checkAndRemoveOrphanedMarkers() {
    // Collect all deltas from paragraphs
    const validDeltas = [];

    // Find all marker paragraphs
    $('.paragraph-type--dme-marker .paragraphs-subform').each(function () {
      const delta = getDeltaFromParagraph($(this));
      if (delta !== null && delta !== undefined) {
        validDeltas.push(delta);
      }
    });

    // Check for orphaned markers and remove them
    $('.dme-marker').each(function () {
      const markerId = $(this).attr('id');
      if (!markerId) return;

      // Extract delta from marker ID
      const delta = parseInt(markerId.replace('dme-marker-', ''), 10);

      // If this delta is not in our valid deltas list, remove it
      if (validDeltas.indexOf(delta) === -1) {
        $(this).remove();
      }
    });
    // Check if we need to show or hide the no markers message
    checkAndHideNoMarkersMessage();
  }

  /**
   * Checks for markers that need to revert to text after a file has been removed.
   */
  function checkAndUpdateMarkers() {
    // Find all markers and check which ones need to revert to text
    $('.dme-marker.dme-marker-icon').each(function () {
      const markerId = $(this).attr('id');
      if (!markerId) return;

      // Extract delta from marker ID
      const delta = markerId.replace('dme-marker-', '');

      // Find the corresponding paragraph by matching the delta
      let foundParagraph = false;
      $('.paragraph-type--dme-marker, .paragraphs-subform').each(function () {
        const paragraphDelta = getDeltaFromParagraph($(this));
        const numericDelta = parseInt(delta, 10);
        if (paragraphDelta === numericDelta) {
          foundParagraph = true;
          const $paragraphItem = $(this);

          // Check if there's a file in the file input
          const $iconInput = $paragraphItem.find(
            '.field--name-field-dme-marker-icon input[type="file"]',
          );
          const hasFile =
            $iconInput.length &&
            $iconInput[0].files &&
            $iconInput[0].files.length > 0;

          // Also check if there's already an uploaded file
          const hasUploadedFile =
            $paragraphItem.find('.field--name-field-dme-marker-icon .file')
              .length > 0;

          if (!hasFile && !hasUploadedFile) {
            // No file, revert to text
            const $titleInput = $paragraphItem.find(
              'input[name*="field_dme_marker_title"]',
            );
            const title = $titleInput.length
              ? $titleInput[0].value
              : `Marker ${delta}`;

            const $wrapper = $(`#dme-marker-${delta}`).find(
              '.dme-marker-wrapper',
            );
            $wrapper.empty();
            $wrapper[0].textContent = title || `Marker ${delta}`;

            // Update classes
            $(`#dme-marker-${delta}`)
              .removeClass('dme-marker-icon')
              .addClass('has-text');
            // Resize font
            resizeFont($(`#dme-marker-${delta}`));
            // Make the marker resizable
            $(`#dme-marker-${delta}`).resizable({
              aspectRatio: false,
              handles: 'all',
              minHeight: 50,
              minWidth: 100,
              containment: '.dme-container-wrapper',
              resize(event, ui) {
                // Update marker font size in real-time
                resizeFont($(this));
              },
            });
          }
        }
      });
    });
  }

  /**
   * Behavior for Draggable Mapper Entity file preview.
   */
  Drupal.behaviors.filePreview = {
    attach(context, settings) {
      once(
        'file-preview',
        '.field--name-field-dme-image input[type="file"]',
        context,
      ).forEach(function (fileInput) {
        $(fileInput).on('change', function () {
          if (this.files && this.files[0]) {
            // Create a file reader to generate a preview from the selected file
            const reader = new FileReader();

            reader.onload = function (e) {
              // Generate preview markup client-side
              const previewMarkup =
                `<div class="dme-image-wrapper">` +
                `<img src="${e.target.result}" alt="Map Image" />` +
                `</div>`;

              // Update both containers with the new client-side generated preview
              $('.dme-image').html(previewMarkup);
            };

            // Read the selected file as a data URL
            reader.readAsDataURL(this.files[0]);
          }
        });
      });
    },
  };

  /**
   * Behavior for Draggable Mapper Entity marker handling.
   */
  Drupal.behaviors.markerHandler = {
    attach(context, settings) {
      // Make markers draggable
      initDraggableMarkers(context);

      // Initialize the Container as Droppable
      $('.dme-container-wrapper').droppable({
        accept: '.dme-marker',
      });

      // Check for existing uploaded marker icons on page load
      once('check-existing-icons', 'body', context).forEach(function () {
        initializeExistingMarkerIcons();
      });

      // Process marker title fields
      once(
        'marker-title-handler',
        '.field--name-field-dme-marker-title input[type="text"]',
        context,
      ).forEach(function (titleInput) {
        // Initialize the marker with current value or default
        updateMarker($(titleInput));
        $(titleInput).on('input', function () {
          updateMarker($(this));
        });
      });

      // Process marker icon fields
      once(
        'marker-icon-handler',
        '.field--name-field-dme-marker-icon input[type="file"]',
        context,
      ).forEach(function (iconInput) {
        $(iconInput).on('change', function () {
          updateMarkerIcon($(this));
        });
      });

      once(
        'paragraph-operations',
        '.field--name-field-dme-marker',
        context,
      ).forEach(function (paragraphField) {
        // Set up a MutationObserver to detect when paragraphs are removed
        const observer = new MutationObserver(function (mutations) {
          // Check if we need to remove markers
          checkAndRemoveOrphanedMarkers();
        });

        // Observe the paragraph container for changes to its children
        observer.observe(paragraphField, {
          childList: true,
          subtree: true,
        });
      });
    },
  };

  /**
   * Behavior for update markers on icon removal.
   */
  Drupal.behaviors.dmeMarkerFileRemovalObserver = {
    attach(context, settings) {
      once(
        'dme-marker-file-removal-observer',
        '.field--name-field-dme-marker-icon',
        context,
      ).forEach(function (element) {
        const observer = new MutationObserver(function (
          mutationsList,
          observer,
        ) {
          // After each mutation, check if the container has any '.file' element.
          const $container = $(element);
          const fileCount = $container.find('.file').length;

          // If no files exist inside the container, trigger marker update.
          if (fileCount === 0) {
            setTimeout(checkAndUpdateMarkers, 500);
          }
        });

        // Observe changes within this container.
        observer.observe(element, { childList: true, subtree: true });
      });
    },
  };

  /**
   * Behavior to manage the unmapped wrapper visibility based on image presence.
   */
  Drupal.behaviors.draggableMapperWrapperManager = {
    attach(context, settings) {
      $(once('wrapper-manager', 'body', context)).each(function () {
        function updateWrapperVisibility() {
          // Target the unmapped wrapper container
          const $wrapper = $('#dme-unmapped-wrapper');
          // Target the unmapped wrapper description
          const $description = $('.form-item__description');

          // Check if image exists using multiple methods
          const imgElements = $('.field--name-field-dme-image img');
          const imgInputs = $(
            'input[name^="field_dme_image"][name$="[fids]"][value!=""]',
          );
          const hasImage = imgElements.length > 0 || imgInputs.length > 0;

          // Update wrapper visibility based on image presence
          if (hasImage) {
            $wrapper.removeClass('empty-container');
            $description.removeClass('empty-container');
          }
        }

        // Run immediately and after any AJAX completes
        updateWrapperVisibility();

        // After (Drupal AJAX events)
        once('ajax-wrapper-update', document).forEach(() => {
          $(document).on('dialog:aftercreate', updateWrapperVisibility);
        });
      });
    },
  };

  /**
   * Behavior to make Mapped Marker resizable.
   */
  Drupal.behaviors.resizableMarker = {
    attach(context, settings) {
      once('resizable-marker', '.dme-mapped-marker', context).forEach(
        function (marker) {
          // Make the marker resizable
          makeMarkerResizable($(marker));

          // Resize marker font
          resizeFont($(marker));
        },
      );

      // Ensure resizable property is maintained after drag operations
      $(document).on('stop', '.dme-mapped-marker', function () {
        // Re-initialize resizable if needed
        if (!$(this).hasClass('ui-resizable')) {
          $(this).resizable({
            aspectRatio: !!$(this).hasClass('dme-marker-icon'),
            handles: 'all',
            containment: '.dme-container-wrapper',
            minWidth: 100,
          });
        }
      });
    },
  };
})(jQuery, Drupal, drupalSettings, once);
