<?php

namespace Drupal\draggableviews_demo\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Link;
use Drupal\Core\Url;

/**
 * Controller for the Draggableviews Demo landing page.
 */
class DemoLandingPageController extends ControllerBase {

  /**
   * Displays the demo landing page.
   *
   * @return array
   *   A render array.
   */
  public function content() {
    $build = [];

    // Introduction
    $build['intro'] = [
      '#type' => 'markup',
      '#markup' => '<div class="draggableviews-demo-intro">
        <h2>' . $this->t('Welcome to the Draggableviews Demo') . '</h2>
        <p>' . $this->t('This demo showcases two different methods for storing drag-and-drop ordering in Drupal:') . '</p>
      </div>',
    ];

    // Native Handler Demo section
    $build['native_demo'] = [
      '#type' => 'details',
      '#title' => $this->t('Native Handler Demo'),
      '#open' => TRUE,
      '#attributes' => ['class' => ['draggableviews-demo-section']],
    ];

    $build['native_demo']['description'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('Uses the internal <code>draggableviews_structure</code> database table to store ordering. This is the default and most efficient method.') . '</p>',
    ];

    $build['native_demo']['features'] = [
      '#theme' => 'item_list',
      '#title' => $this->t('Features:'),
      '#items' => [
        $this->t('Lightweight storage mechanism'),
        $this->t('Works across all entity types'),
        $this->t('Order maintained separately from entities'),
        $this->t('No field configuration required'),
      ],
    ];

    $build['native_demo']['links'] = [
      '#theme' => 'item_list',
      '#title' => $this->t('Try it:'),
      '#items' => [
        Link::fromTextAndUrl($this->t('View ordered content'), Url::fromUri('internal:/draggableviews-demo/native')),
        Link::fromTextAndUrl($this->t('Reorder items (drag & drop)'), Url::fromUri('internal:/draggableviews-demo/native/order')),
      ],
    ];

    // Field API Handler Demo section
    $build['weight_demo'] = [
      '#type' => 'details',
      '#title' => $this->t('Field API Handler Demo'),
      '#open' => TRUE,
      '#attributes' => ['class' => ['draggableviews-demo-section']],
    ];

    $build['weight_demo']['description'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('Uses the <code>field_demo_weight</code> integer field on each node to store ordering. Weight values are stored directly on the entity.') . '</p>',
    ];

    $build['weight_demo']['features'] = [
      '#theme' => 'item_list',
      '#title' => $this->t('Features:'),
      '#items' => [
        $this->t('Weight values accessible in templates: <code>{{ node.field_demo_weight.value }}</code>'),
        $this->t('Weight exports/imports with entity data'),
        $this->t('Useful for displaying position numbers (e.g., "Tip #3")'),
        $this->t('Weight can be queried or filtered like any field'),
      ],
    ];

    $build['weight_demo']['links'] = [
      '#theme' => 'item_list',
      '#title' => $this->t('Try it:'),
      '#items' => [
        Link::fromTextAndUrl($this->t('View ordered content'), Url::fromUri('internal:/draggableviews-demo/weight')),
        Link::fromTextAndUrl($this->t('Reorder items (drag & drop)'), Url::fromUri('internal:/draggableviews-demo/weight/order')),
      ],
    ];

    // Comparison section
    $build['comparison'] = [
      '#type' => 'details',
      '#title' => $this->t('Comparison'),
      '#open' => FALSE,
    ];

    $build['comparison']['table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Feature'),
        $this->t('Native Handler'),
        $this->t('Field API Handler'),
      ],
      '#rows' => [
        [
          $this->t('Storage'),
          $this->t('draggableviews_structure table'),
          $this->t('Entity field (field_demo_weight)'),
        ],
        [
          $this->t('Configuration'),
          $this->t('No field setup needed'),
          $this->t('Requires integer field'),
        ],
        [
          $this->t('Template Access'),
          $this->t('Not directly accessible'),
          $this->t('{{ node.field_demo_weight.value }}'),
        ],
        [
          $this->t('Export/Import'),
          $this->t('Separate from content'),
          $this->t('Included with entity data'),
        ],
        [
          $this->t('Performance'),
          $this->t('Slightly faster'),
          $this->t('Requires field query'),
        ],
        [
          $this->t('Use Case'),
          $this->t('General ordering'),
          $this->t('When weight needs to be displayed/queried'),
        ],
      ],
    ];

    // Additional resources
    $build['resources'] = [
      '#type' => 'details',
      '#title' => $this->t('Additional Resources'),
      '#open' => FALSE,
    ];

    $build['resources']['links'] = [
      '#theme' => 'item_list',
      '#items' => [
        Link::fromTextAndUrl($this->t('Create demo content'), Url::fromUri('internal:/node/add/draggableviews_demo')),
        Link::fromTextAndUrl($this->t('Manage content'), Url::fromUri('internal:/admin/content')),
        Link::fromTextAndUrl($this->t('Configure Native view'), Url::fromUri('internal:/admin/structure/views/view/draggableviews_native_demo')),
        Link::fromTextAndUrl($this->t('Configure Field API view'), Url::fromUri('internal:/admin/structure/views/view/draggableviews_weight_demo')),
        Link::fromTextAndUrl($this->t('DraggableViews project page'), Url::fromUri('https://www.drupal.org/project/draggableviews')),
      ],
    ];

    // Notes
    $build['notes'] = [
      '#type' => 'markup',
      '#markup' => '<div class="draggableviews-demo-notes">
        <h3>' . $this->t('Important Notes') . '</h3>
        <ul>
          <li>' . $this->t('Both demos maintain <strong>independent orderings</strong> - changing one does not affect the other.') . '</li>
          <li>' . $this->t('You need the <em>"access draggableviews"</em> permission to access the order pages.') . '</li>
          <li>' . $this->t('The Field API demo shows the weight value on each node for demonstration purposes.') . '</li>
          <li>' . $this->t('Try creating additional content and reordering in both demos to see the differences.') . '</li>
        </ul>
      </div>',
    ];

    // Add some basic styling
    $build['#attached']['library'][] = 'draggableviews_demo/demo-styles';

    return $build;
  }

}

