<?php

namespace Drupal\draggableviews;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\draggableviews\Handler\FieldApiHandler;
use Drupal\draggableviews\Handler\NativeHandler;

/**
 * Factory for creating draggableviews handlers.
 */
class DraggableViewsHandlerFactory {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a new DraggableViewsHandlerFactory.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(Connection $database, EntityTypeManagerInterface $entity_type_manager, MessengerInterface $messenger) {
    $this->database = $database;
    $this->entityTypeManager = $entity_type_manager;
    $this->messenger = $messenger;
  }

  /**
   * Get a handler by type.
   *
   * @param string $type
   *   The handler type ('native' or 'fieldapi').
   *
   * @return \Drupal\draggableviews\DraggableViewsHandlerInterface
   *   The handler instance.
   *
   * @throws \InvalidArgumentException
   *   When an invalid handler type is requested.
   */
  public function getHandler($type) {
    switch ($type) {
      case 'native':
        return new NativeHandler($this->database, $this->messenger);

      case 'fieldapi':
        return new FieldApiHandler($this->entityTypeManager, $this->messenger);

      default:
        throw new \InvalidArgumentException("Invalid handler type: {$type}");
    }
  }

}

