<?php

namespace Drupal\draggableviews\Handler;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\draggableviews\DraggableViewsHandlerInterface;
use Drupal\views\ViewExecutable;

/**
 * Field API handler for draggableviews.
 *
 * This handler saves weights to an entity field.
 */
class FieldApiHandler implements DraggableViewsHandlerInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a new FieldApiHandler.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, MessengerInterface $messenger) {
    $this->entityTypeManager = $entity_type_manager;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public function getWeight(ViewExecutable $view, $index) {
    $field_options = $view->field['draggableviews']->options ?? [];
    $weight_field = $field_options['draggable_views_weight_field'] ?? '';

    if (empty($weight_field)) {
      return 0;
    }

    $row = $view->result[$index];
    // Try to get the weight from the field.
    if (isset($row->{'_entity'}) && $row->_entity->hasField($weight_field)) {
      $value = $row->_entity->get($weight_field)->value;
      return $value ?? 0;
    }

    return 0;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array &$form, FormStateInterface $form_state, ViewExecutable $view) {
    $field_options = $view->field['draggableviews']->options ?? [];
    $field_id = $field_options['id'] ?? 'draggableviews';
    $weight_field = $field_options['draggable_views_weight_field'] ?? '';

    $this->saveWithConfig($form, $form_state, $view, $field_id, $weight_field);
  }

  /**
   * Save with explicit configuration.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param \Drupal\views\ViewExecutable $view
   *   The view.
   * @param string $field_id
   *   The draggableviews field ID.
   * @param string $weight_field
   *   The weight field name.
   */
  public function saveWithConfig(array &$form, FormStateInterface $form_state, ViewExecutable $view, $field_id, $weight_field) {
    if (empty($weight_field)) {
      $this->messenger->addError(t('No weight field configured.'));
      return;
    }

    // Get values from user input (this is where tabledrag stores the data).
    $user_input = $form_state->getUserInput();
    $values = $user_input[$field_id] ?? NULL;

    if (empty($values) || !is_array($values)) {
      $this->messenger->addWarning(t('No values to save.'));
      return;
    }

    // Sort items by weight before saving.
    uasort($values, function ($a, $b) {
      return ($a['weight'] ?? 0) <=> ($b['weight'] ?? 0);
    });

    $entity_type_id = $view->getBaseEntityType()->id();
    $entity_storage = $this->entityTypeManager->getStorage($entity_type_id);

    // Save each entity with its new weight, starting from 1.
    $saved_count = 0;
    $normalized_weight = 1;
    foreach ($values as $item) {
      if (empty($item['id']) || !isset($item['weight'])) {
        continue;
      }

      try {
        $entity = $entity_storage->load($item['id']);
        if ($entity && $entity->hasField($weight_field)) {
          // Use normalized weight starting from 1 instead of raw tabledrag weight.
          $entity->set($weight_field, $normalized_weight);
          $entity->save();
          $saved_count++;
          $normalized_weight++;
        }
      }
      catch (\Exception $e) {
        $this->messenger->addError(t('Failed to save weight for entity @id: @message', [
          '@id' => $item['id'],
          '@message' => $e->getMessage(),
        ]));
      }
    }

    if ($saved_count > 0) {
      $this->messenger->addStatus(t('Order has been saved. Updated @count items.', ['@count' => $saved_count]));
    }
    else {
      $this->messenger->addWarning(t('No items were saved.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(array $form, FormStateInterface $form_state, array $options) {
    $view = $form_state->get('view');

    if (!$view instanceof ViewExecutable) {
      return $form;
    }

    $weight_field_options = $this->getAvailableWeightFields($view);

    if (empty($weight_field_options)) {
      $form['field_warning'] = [
        '#markup' => '<div class="messages messages--warning">' . t('No integer fields available. Add an integer field to your entity to use Field API ordering.') . '</div>',
      ];
    }

    $form['draggable_views_weight_field'] = [
      '#type' => 'select',
      '#title' => t('Weight field'),
      '#options' => $weight_field_options,
      '#default_value' => $options['draggable_views_weight_field'] ?? '',
      '#description' => t('Select the integer field to use for storing the weight.'),
      '#empty_option' => t('- Select a weight field -'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultOptions() {
    return [
      'draggable_views_weight_field' => '',
    ];
  }

  /**
   * Get available weight fields from the view.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view executable object.
   *
   * @return array
   *   Array of available field options.
   */
  protected function getAvailableWeightFields(ViewExecutable $view) {
    $options = [];

    if (!$view->getBaseEntityType()) {
      return $options;
    }

    $entity_type_id = $view->getBaseEntityType()->id();

    // Get all bundles used in the view.
    $bundles = [];
    if (isset($view->filter['type']) && !empty($view->filter['type']->value)) {
      $bundles = is_array($view->filter['type']->value) ? $view->filter['type']->value : [$view->filter['type']->value];
    }

    // If no specific bundle filter, get all bundles for this entity type.
    if (empty($bundles)) {
      $bundle_info = \Drupal::service('entity_type.bundle.info')->getBundleInfo($entity_type_id);
      $bundles = array_keys($bundle_info);
    }

    // Get integer fields available across all bundles.
    $field_manager = \Drupal::service('entity_field.manager');
    $field_intersection = [];

    foreach ($bundles as $bundle) {
      $field_definitions = $field_manager->getFieldDefinitions($entity_type_id, $bundle);
      $bundle_fields = [];

      foreach ($field_definitions as $field_name => $field_definition) {
        if ($field_definition->getType() === 'integer') {
          $bundle_fields[] = $field_name;
          $options[$field_name] = $field_definition->getLabel() . ' (' . $field_name . ')';
        }
      }

      if (empty($field_intersection)) {
        $field_intersection = $bundle_fields;
      }
      else {
        $field_intersection = array_intersect($field_intersection, $bundle_fields);
      }
    }

    return $options;
  }

}

